/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.InetAddress;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.videolan.vlma.common.adapters.IVlAdapter;
import org.videolan.vlma.common.exceptions.AdapterAlreadyExistsException;
import org.videolan.vlma.common.exceptions.AdapterDoesNotExistException;
import org.videolan.vlma.common.exceptions.AdapterParameterDoesNotExistException;
import org.videolan.vlma.common.exceptions.MediaDoesNotExistException;
import org.videolan.vlma.common.exceptions.SatelliteAlreadyExistsException;
import org.videolan.vlma.common.exceptions.SatelliteDoesNotExistException;
import org.videolan.vlma.common.exceptions.ServerAlreadyExistsException;
import org.videolan.vlma.common.exceptions.ServerDoesNotExistException;
import org.videolan.vlma.common.medias.IVlMedia;
import org.videolan.vlma.common.medias.VlSatellite;
import org.videolan.vlma.common.orders.VlOrder;
import org.videolan.vlma.common.programs.IVlProgram;
import org.videolan.vlma.common.programs.VlProgram;

/**
 * A Data interface.
 * This interface provides a simple way to control the application data (servers, 
 * adapters, media, programs, etc.). 
 * 
 * @author SylV
 * @version 1.0
 */
public interface IVlData {

    // Configuration
    /**
     * Returns the VLMa configuration. 
     *
     * @see Configuration
     * @return the VLMa configuration
     */
    public Configuration getConfiguration();
    
    // Satellites
    /**
     * Returns the list of the satellites known by the data interface. 
     *
     * @see VlSatellite
     * @return la liste des satellites
     */
    public List<VlSatellite> getSatellites();

    /**
     * Returns a satellite according to its hashcode.
     * 
     * @param satellite the hashcode of the satellite to retrieve
     * @return the satellite
     * @throws SatelliteDoesNotExistException
     *             No satellite has the specified hashcode.
     */
    public VlSatellite getSatellite(int satellite)
            throws SatelliteDoesNotExistException;

    /**
     * Clears the satellite coverage zones.
     * 
     * @param satellite the satellite hashcode
     * @throws SatelliteDoesNotExistException
     *             No satellite has the specified hashcode.
     */
    public void clearSatelliteCoverages(int satellite)
            throws SatelliteDoesNotExistException;

    /**
     * Adds a coverage zone to a satellite.
     * 
     * @param satellite the satellite hashcode
     * @param coverage the satellite coverage zone
     * @throws SatelliteDoesNotExistException
     *             No satellite has the specified hashcode.
     */
    public void addSatelliteCoverage(int satellite, String coverage)
            throws SatelliteDoesNotExistException;

    /**
     * Adds a satellite.
     * 
     * @param name the satellite name
     * @return the hashcode of the newly created satellite
     * @throws SatelliteAlreadyExistsException
     *             Aoother satellite already has this name.
     */
    public int addSatellite(String name) throws SatelliteAlreadyExistsException;

    /**
     * Returns the coverage zones of a satellite.
     * 
     * @param satellite the satellite hashcode
     * @return the coverage zones of the satellite
     * @throws SatelliteDoesNotExistException
     *             No satellite has the specified hashcode.
     */
    public List<String> getSatelliteCoverages(int satellite)
            throws SatelliteDoesNotExistException;

    /**
     * Deletes a satellite.
     * 
     * @param satellite the satellite hashcode
     * @throws SatelliteDoesNotExistException
     *             No satellite has the specified hashcode.
     */
    public void removeSatellite(int satellite)
            throws SatelliteDoesNotExistException;

    // Serveurs
    /**
     * Returns the satellites managed by the data interface.
     * 
     * @see VlServer
     * @return the server list
     */
    public List<VlServer> getServers();

    /**
     * Returns a satellite according to its hashcode.
     * 
     * @see VlServer
     * @see VlServer#hashCode()
     * @param server the server hashcode
     * @return the server
     * @throws ServerDoesNotExistException
     *             No server has the specified hashcode.
     */
    public VlServer getServer(int server) throws ServerDoesNotExistException;

    /**
     * Returns a satellite according to its name.
     *
     * @param server the server name
     * @return the server
     * @throws ServerDoesNotExistException
     *             No server has the specified name.
     */
    public VlServer getServer(String server) throws ServerDoesNotExistException;

    /**
     * Creates a server
     * 
     * @see VlServer
     * @see VlServer#hashCode()
     * @param name the server name
     * @return the hashcode of the newly created server
     * @throws ServerAlreadyExistsException
     *             Another server has already that name.
     */
    public int addServer(String name) throws ServerAlreadyExistsException;

    /**
     * Sets the IP address of a server
     * 
     * @param server the server hashcode
     * @param newIp the new IP of this server
     * @throws ServerDoesNotExistException
     *             No server has the specified hashcode.
     */
    public void setServerIp(int server, InetAddress newIp)
            throws ServerDoesNotExistException;

    /**
     * Deletes a server.
     * 
     * @param server the server hashcode
     * @throws ServerDoesNotExistException
     *             No server has the specified hashcode.
     */
    public void removeServer(int server) throws ServerDoesNotExistException;

    // Périphériques
    /**
     * Returns the adapters of a server.
     * 
     * @param server the server hashcode
     * @return a map of the adapters of the server
     */
    public Map<String, IVlAdapter> getAdapters(int server)
            throws ServerDoesNotExistException;

    /**
     * Returns an adapter according to its name and the server its belongs.
     * 
     * @param server the server hashcode
     * @param name the adapter name
     * @return the adapter
     * @throws ServerDoesNotExistException
     *             No server has the specified hashcode.
     * @throws AdapterDoesNotExistException
     *             No adapter has the specified hashcode.
     */
    public IVlAdapter getAdapter(int server, String name)
            throws ServerDoesNotExistException, AdapterDoesNotExistException;

    /**
     * Changes an adapter name.
     * 
     * @param server the server hashcode
     * @param name the server name
     * @param newName the new adapter name
     * @throws AdapterDoesNotExistException
     *             No adapter has the specified hashcode.
     * @throws AdapterAlreadyExistsException
     *             There already is an adapter with the same name.
     */
    public void setAdapterName(int server, String name, String newName)
            throws AdapterDoesNotExistException, AdapterAlreadyExistsException;

    /**
     * Changes a parameter of an adapter.
     * 
     * @param server the server hashcode
     * @param name the adapter name
     * @param parameter the name of the parameter to modify
     * @param value the new parameter value
     * @throws AdapterDoesNotExistException
     *             No adapter has the specified hashcode.
     * @throws AdapterParameterDoesNotExistException
     *             The adapter has no such parameter.
     */
    public void setAdapterParameter(int server, String name, String parameter,
            String value) throws AdapterDoesNotExistException,
            AdapterParameterDoesNotExistException;

    /**
     * Adds an adapter to a server
     * 
     * @param server the server adapter
     * @param name the adapter name
     * @param type the adapter type
     * @throws AdapterAlreadyExistsException
     *             There already is an adapter with this name.
     * @throws IllegalAccessException
     * @throws InstantiationException
     *             Impossible to instanciate the class.
     * @throws ClassNotFoundException
     *             There is no such adapter type.
     */
    public void addAdapter(int server, String name, String type)
            throws AdapterAlreadyExistsException, IllegalAccessException,
            InstantiationException, ClassNotFoundException;

    /**
     * Deletes an adapter from a server.
     * 
     * @param server the server hashcode
     * @param name the adapter name
     * @throws AdapterDoesNotExistException
     *             No server has the specified hashcode.
     */
    public void removeAdapter(int server, String name)
            throws AdapterDoesNotExistException;

    // Médias
    /**
     * Returns the media handled by the media interface.
     * 
     * @return the media list
     */
    public List<IVlMedia> getMedias();

    /**
     * Returns a media according to its hashcode.
     * 
     * @param media the media hashcode
     * @return the media
     */
    public IVlMedia getMedia(int media);

    /**
     * Adds a media to the list of handled media.
     * 
     * @param media the media to add
     * @return the media hashcode
     */
    public int addMedia(IVlMedia media);

    
    /**
     * Sorts the handled media.
     */
    public void sortMedias();
    
    /** Adds the path to file to the VlFilesChannels
     * 
     * @param filesChannel the VlFilesChannels
     * @param files the file path
     */    
    public void setFilesChannelFilesList(int filesChannel, String files);
    
    /** Adds the server to the VlFilesChannels
     * 
     * @param filesChannel the VlFilesChannel
     * @param server the server
     */    
    public void setFilesChannelServer(int filesChannel, String server);
    
    /**
     * Updates the program associated to a media.
     * 
     * @param media the media hashcode
     * @param program the new program
     * @throws MediaDoesNotExistException
     *             No media has the specified hashcode.
     */
    public void updateMediaProgram(int media, IVlProgram program)
            throws MediaDoesNotExistException;

    /**
     * Creates the program of a media.
     * 
     * @param media the media hashcode
     * @return the program
     * @throws MediaDoesNotExistException
     *             No media has the specified hashcode.
     */
    public VlProgram addMediaProgram(int media)
            throws MediaDoesNotExistException;

    /**
     * Deletes a media
     * 
     * @param media the media hashcode
     * @throws MediaDoesNotExistException
     *             No media has the specified hashcode.
     */
    public void removeMedia(int media) throws MediaDoesNotExistException;

    // Mise à jour des chaînes satellites
    /**
     * Updates the list of a satellite channels using an URL like 
     * http://www.satcodx2.com/_data/0192.txt.
     * 
     * @param source the url to use to update
     * @throws IOException 
     *             Impossible to fetch the channel list. 
     */
    public void updateSatChannels(URL source) throws IOException;

    // Ordres
    /**
     * Returns the list of the given orders.
     * 
     * @return the order list
     * @see VlOrder
     */
    public HashMap<Integer, List<VlOrder>> getOrders();

    /**
     * Executes orders.
     * 
     * @see IVlOrderGiver
     */
    public void giveOrders();

    
    /**
     * Gets the VlOrderGiver of the deomon.
     * 
     * @return the VlOrderGiver of the deamon
     */
    public IVlOrderGiver getOrderGiver();
    
    // Sauvegarde et restauration des paramètres
    /**
     * Saves the configuration to a file.
     * 
     * @throws FileNotFoundException
     *             Impossible to find the configuration file.
     * @throws IOException
     *             I/O error.
     */
    public void saveToDisk() throws FileNotFoundException, IOException;

    /**
     * Loads the configuration from a file.
     * 
     * @throws FileNotFoundException 
     *             Impossible to find the configuration file.
     * @throws IOException
     *             I/O error.
     */
    public void loadFromDisk() throws FileNotFoundException, IOException;
    
    /* Starts the punctual monitoring controls */
    
    /**
     * Starts monitoring the execution of the diffusion orders.
     */
    public void startOrderMonitoring();
    
    /**
     * Checks if all the VLC of the servers are up.
     */
    public void startCheckAllVLCs();
}
