/*
 * (C) Copyright 1997-2002 ELTEC Elektronik AG
 * Frank Gottschling <fgottschling@eltec.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * smiLynxEM.c
 *
 * Silicon Motion graphic interface for sm810/sm710/sm712 accelerator
 *
 * modification history
 * --------------------
 * 04-18-2002 Rewritten for U-Boot <fgottschling@eltec.de>.
 *
 */

#include <common.h>

#if defined(CONFIG_VIDEO_SMI_LYNXEM)

#include <pci.h>
#include <video_fb.h>

/*
 * Export Graphic Device
 */
GraphicDevice smi;

/*
 * SMI 710/712 have 4MB internal RAM; SMI 810 2MB internal + 2MB external
 */
#define VIDEO_MEM_SIZE  0x400000

/*
 * Supported video modes for SMI Lynx E/EM/EM+
 */
#define VIDEO_MODES             7
#define DUAL_800_600            0   /* SMI710:VGA1:75Hz     (pitch=1600) */
                                    /*        VGA2:60/120Hz (pitch=1600) */
                                    /* SMI810:VGA1:75Hz     (pitch=1600) */
                                    /*        VGA2:75Hz     (pitch=1600) */
#define DUAL_1024_768           1   /* VGA1:75Hz VGA2:73Hz (pitch=2048)  */
#define SINGLE_800_600          2   /* VGA1:75Hz (pitch=800)             */
#define SINGLE_1024_768         3   /* VGA1:75Hz (pitch=1024)            */
#define SINGLE_1280_1024        4   /* VGA1:75Hz (pitch=1280)            */
#define TV_MODE_CCIR            5   /* VGA1:50Hz (h=720;v=576;pitch=720) */
#define TV_MODE_EIA             6   /* VGA1:60Hz (h=720;v=484;pitch=720) */


/*
 * ISA mapped regs
 */
#define SMI_INDX_C4             (pGD->isaBase + 0x03c4)    /* index reg */
#define SMI_DATA_C5             (pGD->isaBase + 0x03c5)    /* data reg */
#define SMI_INDX_D4             (pGD->isaBase + 0x03d4)    /* index reg */
#define SMI_DATA_D5             (pGD->isaBase + 0x03d5)    /* data reg */
#define SMI_INDX_CE             (pGD->isaBase + 0x03ce)    /* index reg */
#define SMI_DATA_CF             (pGD->isaBase + 0x03cf)    /* data reg */
#define SMI_LOCK_REG            (pGD->isaBase + 0x03c3)    /* unlock/lock ext crt reg */
#define SMI_MISC_REG            (pGD->isaBase + 0x03c2)    /* misc reg */
#define SMI_LUT_MASK            (pGD->isaBase + 0x03c6)    /* lut mask reg */
#define SMI_LUT_START           (pGD->isaBase + 0x03c8)    /* lut start index */
#define SMI_LUT_RGB             (pGD->isaBase + 0x03c9)    /* lut colors auto incr.*/


/*
 * Video processor control
*/
typedef struct {
    unsigned int   control;
    unsigned int   colorKey;
    unsigned int   colorKeyMask;
    unsigned int   start;
    unsigned short offset;
    unsigned short width;
    unsigned int   fifoPrio;
    unsigned int   fifoERL;
    unsigned int   YUVtoRGB;
} SmiVideoProc;

/*
 * Video window control
 */
typedef struct {
    unsigned short top;
    unsigned short left;
    unsigned short bottom;
    unsigned short right;
    unsigned int   srcStart;
    unsigned short width;
    unsigned short offset;
    unsigned char  hStretch;
    unsigned char  vStretch;
} SmiVideoWin;

/*
 * Capture port control
 */
typedef struct {
    unsigned int   control;
    unsigned short topClip;
    unsigned short leftClip;
    unsigned short srcHeight;
    unsigned short srcWidth;
    unsigned int   srcBufStart1;
    unsigned int   srcBufStart2;
    unsigned short srcOffset;
    unsigned short fifoControl;
} SmiCapturePort;



/*
 * Register values for common video modes
 */
static char SMI_SCR[22] = {
    /* all modes */
    0x10, 0xff, 0x11, 0xff, 0x12, 0xff, 0x13, 0xff, 0x14, 0x00, 0x15, 0x90,
    0x16, 0x10, 0x17, 0x2c, 0x18, 0xb1, 0x19, 0x20, 0x1a, 0x01
};

static char SMI_EXT_CRT[VIDEO_MODES][24] = {
    { /* DUAL_800_600_8 */
    0x30, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00, 0x34, 0x00, 0x35, 0x00,
    0x36, 0x00, 0x3b, 0x00, 0x3c, 0x00, 0x3d, 0x00, 0x3e, 0x00, 0x3f, 0x00
    },
    { /* DUAL_1024_768_8 */
    0x30, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00, 0x34, 0x00, 0x35, 0x00,
    0x36, 0x00, 0x3b, 0x00, 0x3c, 0x00, 0x3d, 0x00, 0x3e, 0x00, 0x3f, 0x00
    },
    { /* SINGLE_800_600_8 */
    0x30, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00, 0x34, 0x00, 0x35, 0x00,
    0x36, 0x00, 0x3b, 0x00, 0x3c, 0x00, 0x3d, 0x00, 0x3e, 0x00, 0x3f, 0x00
    },
    { /* SINGLE_1024_768_8 */
    0x30, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00, 0x34, 0x00, 0x35, 0x00,
    0x36, 0x00, 0x3b, 0x00, 0x3c, 0x00, 0x3d, 0x00, 0x3e, 0x00, 0x3f, 0x00
    },
    { /* SINGLE_1280_1024_8 */
    0x30, 0x09, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00, 0x34, 0x00, 0x35, 0x00,
    0x36, 0x00, 0x3b, 0x00, 0x3c, 0x00, 0x3d, 0x00, 0x3e, 0x00, 0x3f, 0x00
    },
    { /* TV_MODE_CCIR */
    0x30, 0x80, 0x31, 0x2b, 0x32, 0x06, 0x33, 0x01, 0x34, 0x26, 0x35, 0x88,
    0x36, 0x02, 0x38, 0x11, 0x39, 0x11, 0x3a, 0x20, 0x3e, 0xa3, 0x3f, 0x00
    },
    { /* TV_MODE_EIA */
    0x30, 0x80, 0x31, 0x2b, 0x32, 0x06, 0x33, 0x00, 0x34, 0xf8, 0x35, 0x88,
    0x36, 0x02, 0x38, 0x11, 0x39, 0x11, 0x3a, 0x20, 0x3e, 0xa3, 0x3f, 0x00
    },
};

static char SMI_CRTCR[VIDEO_MODES][50] = {
    { /* DUAL_800_600_8 */
    0x00, 0x7f, 0x01, 0x63, 0x02, 0x63, 0x03, 0x00, 0x04, 0x68, 0x05, 0x12,
    0x06, 0x6f, 0x07, 0xf0, 0x08, 0x00, 0x09, 0x60, 0x0a, 0x00, 0x0b, 0x00,
    0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x59, 0x11, 0x2c,
    0x12, 0x57, 0x13, 0x64, 0x14, 0x40, 0x15, 0x57, 0x16, 0x00, 0x17, 0xe3,
    0x18, 0xff
    },
    { /* DUAL_1024_768_8 */
    0x00, 0x9f, 0x01, 0x7f, 0x02, 0x7f, 0x03, 0x00, 0x04, 0x82, 0x05, 0x0e,
    0x06, 0x1e, 0x07, 0xf5, 0x08, 0x00, 0x09, 0x60, 0x0a, 0x00, 0x0b, 0x00,
    0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x01, 0x11, 0x24,
    0x12, 0xff, 0x13, 0x80, 0x14, 0x40, 0x15, 0xff, 0x16, 0x00, 0x17, 0xe3,
    0x18, 0xff
    },
    { /* SINGLE_800_600_8 */
    0x00, 0x7f, 0x01, 0x63, 0x02, 0x63, 0x03, 0x00, 0x04, 0x68, 0x05, 0x12,
    0x06, 0x6f, 0x07, 0xf0, 0x08, 0x00, 0x09, 0x60, 0x0a, 0x00, 0x0b, 0x00,
    0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x59, 0x11, 0x2c,
    0x12, 0x57, 0x13, 0x32, 0x14, 0x40, 0x15, 0x57, 0x16, 0x00, 0x17, 0xe3,
    0x18, 0xff
    },
    { /* SINGLE_1024_768_8 */
    0x00, 0x9f, 0x01, 0x7f, 0x02, 0x7f, 0x03, 0x00, 0x04, 0x82, 0x05, 0x0e,
    0x06, 0x1e, 0x07, 0xf5, 0x08, 0x00, 0x09, 0x60, 0x0a, 0x00, 0x0b, 0x00,
    0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x01, 0x11, 0x24,
    0x12, 0xff, 0x13, 0x40, 0x14, 0x40, 0x15, 0xff, 0x16, 0x00, 0x17, 0xe3,
    0x18, 0xff
    },
    { /* SINGLE_1280_1024_8 */
    0x00, 0xce, 0x01, 0x9f, 0x02, 0x9f, 0x03, 0x00, 0x04, 0xa2, 0x05, 0x12,
    0x06, 0x2a, 0x07, 0x5a, 0x08, 0x00, 0x09, 0x60, 0x0a, 0x00, 0x0b, 0x00,
    0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x01, 0x11, 0x23,
    0x12, 0xff, 0x13, 0x50, 0x14, 0x40, 0x15, 0xff, 0x16, 0x00, 0x17, 0xe3,
    0x18, 0xff
    },
    { /* TV_MODE_CCIR */
    0x00, 0x00, 0x01, 0x59, 0x02, 0x63, 0x03, 0x00, 0x04, 0x69, 0x05, 0x10,
    0x06, 0x72, 0x07, 0xf0, 0x08, 0x00, 0x09, 0x60, 0x0a, 0x00, 0x0b, 0x00,
    0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x58, 0x11, 0x2c,
    0x12, 0x57, 0x13, 0x2d, 0x14, 0x40, 0x15, 0x57, 0x16, 0x00, 0x17, 0xe3,
    0x18, 0xff
    },
    { /* TV_MODE_EIA */
    0x00, 0x00, 0x01, 0x59, 0x02, 0x63, 0x03, 0x00, 0x04, 0x69, 0x05, 0x10,
    0x06, 0x72, 0x07, 0xf0, 0x08, 0x00, 0x09, 0x60, 0x0a, 0x00, 0x0b, 0x00,
    0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x58, 0x11, 0x2c,
    0x12, 0x57, 0x13, 0x2d, 0x14, 0x40, 0x15, 0x57, 0x16, 0x00, 0x17, 0xe3,
    0x18, 0xff
    },
};

static char SMI_SEQR[10] = {
    0x00, 0x03, 0x01, 0x01, 0x02, 0x0f, 0x03, 0x03, 0x04, 0x0e
};

static char SMI_PCR[VIDEO_MODES][8] = {
    { /* DUAL_800_600_8 */
    0x20, 0x04, 0x21, 0x20, 0x22, 0x00, 0x23, 0x00
    },
    { /* DUAL_1024_768_8 */
    0x20, 0x04, 0x21, 0x20, 0x22, 0x00, 0x23, 0x00
    },
    { /* SINGLE_800_600_8 */
    0x20, 0x04, 0x21, 0x30, 0x22, 0x02, 0x23, 0x00
    },
    { /* SINGLE_1024_768_8 */
    0x20, 0x04, 0x21, 0x30, 0x22, 0x02, 0x23, 0x00
    },
    { /* SINGLE_1280_1024_8 */
    0x20, 0x04, 0x21, 0x30, 0x22, 0x02, 0x23, 0x00
    },
    { /* TV_MODE_CCIR */
    0x20, 0x04, 0x21, 0x30, 0x22, 0x02, 0x23, 0x00
    },
    { /* TV_MODE_EIA */
    0x20, 0x04, 0x21, 0x30, 0x22, 0x02, 0x23, 0x00
    },
};

static char SMI_MCR[VIDEO_MODES][6] = {
    { /* DUAL_800_600_8 */
    0x60, 0x01, 0x61, 0x00, 0x62, 0x7a
    },
    { /* DUAL_1024_768_8 */
    0x60, 0x01, 0x61, 0x00, 0x62, 0x7a
    },
    { /* SINGLE_800_600_8 */
    0x60, 0x00, 0x61, 0x00, 0x62, 0x34
    },
    { /* SINGLE_1024_768_8 */
    0x60, 0x00, 0x61, 0x00, 0x62, 0xfe
    },
    { /* SINGLE_1280_1024_8 */
    0x60, 0x00, 0x61, 0x00, 0x62, 0xfe
    },
    { /* TV_MODE_CCIR */
    0x60, 0x00, 0x61, 0x00, 0x62, 0x34
    },
    { /* TV_MODE_EIA */
    0x60, 0x00, 0x61, 0x00, 0x62, 0x34
    },
};

static char SMI_CCR[VIDEO_MODES][18] = {
    { /* DUAL_800_600_8 */
    0x65, 0x34, 0x68, 0x50, 0x69, 0x05, 0x6a, 0x53, 0x6b, 0x15, 0x6c, 0x15,
    0x6d, 0x06, 0x6e, 0x3d, 0x6f, 0x12
    },
    { /* DUAL_1024_768_8 */
    0x65, 0x00, 0x68, 0x50, 0x69, 0x06, 0x6a, 0x53, 0x6b, 0x15, 0x6c, 0x0b,
    0x6d, 0x02, 0x6e, 0x0b, 0x6f, 0x02
    },
    { /* SINGLE_800_600_8 */
    0x65, 0x34, 0x68, 0x40, 0x69, 0x03, 0x6a, 0x53, 0x6b, 0x15, 0x6c, 0x15,
    0x6d, 0x06, 0x6e, 0x3d, 0x6f, 0x12
    },
    { /* SINGLE_1024_768_8 */
    0x65, 0x00, 0x68, 0x50, 0x69, 0x03, 0x6a, 0x53, 0x6b, 0x15, 0x6c, 0x0b,
    0x6d, 0x02, 0x6e, 0x0b, 0x6f, 0x02
    },
    { /* SINGLE_1280_1024_8 */
    0x65, 0x00, 0x68, 0x50, 0x69, 0x03, 0x6a, 0x53, 0x6b, 0x15, 0x6c, 0xd9,
    0x6d, 0x17, 0x6e, 0xd9, 0x6f, 0x17
    },
    { /* TV_MODE_CCIR */
    0x65, 0x07, 0x68, 0xc0, 0x69, 0x81, 0x6a, 0x53, 0x6b, 0x15, 0x6c, 0x15,
    0x6d, 0x06, 0x6e, 0x3d, 0x6f, 0x12
    },
    { /* TV_MODE_EIA */
    0x65, 0x07, 0x68, 0xc0, 0x69, 0x81, 0x6a, 0x53, 0x6b, 0x15, 0x6c, 0x15,
    0x6d, 0x06, 0x6e, 0x3d, 0x6f, 0x12
    },
};

static char SMI_SHVGA[VIDEO_MODES][24] = {
    { /* DUAL_800_600_8 */
    0x40, 0x7f, 0x41, 0x63, 0x42, 0x00, 0x43, 0x68, 0x44, 0x12, 0x45, 0x6f,
    0x46, 0x57, 0x47, 0x00, 0x48, 0x59, 0x49, 0x0c, 0x4a, 0xa0, 0x4b, 0x20,
    },
    { /* DUAL_1024_768_8 */
    0x40, 0x9f, 0x41, 0x7f, 0x42, 0x00, 0x43, 0x82, 0x44, 0x0e, 0x45, 0x1e,
    0x46, 0xff, 0x47, 0x00, 0x48, 0x00, 0x49, 0x03, 0x4a, 0xe5, 0x4b, 0x20,
    },
    { /* SINGLE_800_600_8 */
    0x40, 0x7f, 0x41, 0x63, 0x42, 0x00, 0x43, 0x68, 0x44, 0x12, 0x45, 0x6f,
    0x46, 0x57, 0x47, 0x00, 0x48, 0x59, 0x49, 0x0c, 0x4a, 0xa0, 0x4b, 0x20,
    },
    { /* SINGLE_1024_768_8 */
    0x40, 0x9f, 0x41, 0x7f, 0x42, 0x00, 0x43, 0x82, 0x44, 0x0e, 0x45, 0x1e,
    0x46, 0xff, 0x47, 0x00, 0x48, 0x01, 0x49, 0x04, 0x4a, 0xa5, 0x4b, 0x20,
    },
    { /* SINGLE_1280_1024_8 */
    0x40, 0xce, 0x41, 0x9f, 0x42, 0x00, 0x43, 0xa2, 0x44, 0x12, 0x45, 0x2a,
    0x46, 0xff, 0x47, 0x00, 0x48, 0x01, 0x49, 0x03, 0x4a, 0x4a, 0x4b, 0x20,
    },
    { /* TV_MODE_CCIR */
    0x40, 0x6d, 0x41, 0x59, 0x42, 0x00, 0x43, 0x60, 0x44, 0x09, 0x45, 0x38,
    0x46, 0x25, 0x47, 0x05, 0x48, 0x2a, 0x49, 0x00, 0x4a, 0x4d, 0x4b, 0x00,
    },
    { /* TV_MODE_EIA */
    0x40, 0x6d, 0x41, 0x59, 0x42, 0x00, 0x43, 0x60, 0x44, 0x09, 0x45, 0x06,
    0x46, 0xf7, 0x47, 0x05, 0x48, 0xfa, 0x49, 0x00, 0x4a, 0x41, 0x4b, 0x00,
    },
};


static char SMI_GPR[VIDEO_MODES][12] = {
    { /* DUAL_800_600_8 */
    0x70, 0x00, 0x71, 0xa2, 0x72, 0x0f, 0x73, 0x30, 0x74, 0x40, 0x75, 0x00
    },
    { /* DUAL_1024_768_8 */
    0x70, 0x00, 0x71, 0xa2, 0x72, 0x0f, 0x73, 0x30, 0x74, 0x40, 0x75, 0x00
    },
    { /* SINGLE_800_600_8 */
    0x70, 0x00, 0x71, 0xa2, 0x72, 0x0f, 0x73, 0x30, 0x74, 0x40, 0x75, 0x00
    },
    { /* SINGLE_1024_768_8 */
    0x70, 0x00, 0x71, 0xa2, 0x72, 0x0f, 0x73, 0x30, 0x74, 0x40, 0x75, 0x00
    },
    { /* SINGLE_1280_1024_8 */
    0x70, 0x00, 0x71, 0xa2, 0x72, 0x0f, 0x73, 0x30, 0x74, 0x40, 0x75, 0x00
    },
    { /* TV_MODE_CCIR */
    0x70, 0x82, 0x71, 0x8d, 0x72, 0x0c, 0x73, 0x32, 0x74, 0x09, 0x75, 0x28
    },
    { /* TV_MODE_EIA */
    0x70, 0x82, 0x71, 0x8d, 0x72, 0x0c, 0x73, 0x32, 0x74, 0x09, 0x75, 0x28
    },
};

static char SMI_HCR[VIDEO_MODES][22] = {
    { /* DUAL_800_600_8 */
    0x80, 0xff, 0x81, 0x07, 0x82, 0x00, 0x83, 0xff, 0x84, 0xff, 0x88, 0x00,
    0x89, 0x02, 0x8a, 0x80, 0x8b, 0x01, 0x8c, 0xff, 0x8d, 0x00
    },
    { /* DUAL_1024_768_8 */
    0x80, 0xff, 0x81, 0x07, 0x82, 0x00, 0x83, 0xff, 0x84, 0xff, 0x88, 0x00,
    0x89, 0x02, 0x8a, 0x80, 0x8b, 0x01, 0x8c, 0xff, 0x8d, 0x00
    },
    { /* SINGLE_800_600_8 */
    0x80, 0xff, 0x81, 0x07, 0x82, 0x00, 0x83, 0xff, 0x84, 0xff, 0x88, 0x00,
    0x89, 0x02, 0x8a, 0x80, 0x8b, 0x01, 0x8c, 0xff, 0x8d, 0x00
    },
    { /* SINGLE_1024_768_8 */
    0x80, 0xff, 0x81, 0x07, 0x82, 0x00, 0x83, 0xff, 0x84, 0xff, 0x88, 0x00,
    0x89, 0x02, 0x8a, 0x80, 0x8b, 0x01, 0x8c, 0xff, 0x8d, 0x00
    },
    { /* SINGLE_1280_1024_8 */
    0x80, 0xff, 0x81, 0x07, 0x82, 0x00, 0x83, 0xff, 0x84, 0xff, 0x88, 0x00,
    0x89, 0x02, 0x8a, 0x80, 0x8b, 0x01, 0x8c, 0xff, 0x8d, 0x00
    },
    { /* TV_MODE_CCIR */
    0x80, 0xff, 0x81, 0x07, 0x82, 0x00, 0x83, 0xff, 0x84, 0xff, 0x88, 0x00,
    0x89, 0x02, 0x8a, 0x80, 0x8b, 0x01, 0x8c, 0xff, 0x8d, 0x00
    },
    { /* TV_MODE_EIA */
    0x80, 0xff, 0x81, 0x07, 0x82, 0x00, 0x83, 0xff, 0x84, 0xff, 0x88, 0x00,
    0x89, 0x02, 0x8a, 0x80, 0x8b, 0x01, 0x8c, 0xff, 0x8d, 0x00
    },
};

static char SMI_FPR[VIDEO_MODES][88] = {
    { /* DUAL_800_600_8 */
    0x30, 0x36, 0x31, 0x83, 0x32, 0x38, 0x33, 0x00, 0x34, 0x40, 0x3e, 0x03,
    0x3f, 0xff, 0x40, 0x64, 0x41, 0x00, 0x42, 0x00, 0x43, 0x00, 0x44, 0xc8,
    0x45, 0x02, 0x46, 0x00, 0x47, 0xfc, 0x48, 0x20, 0x49, 0x1c, 0x4a, 0x41,
    0x4b, 0xa0, 0x4c, 0x00,
    0x50, 0x04, 0x51, 0x48, 0x52, 0x83, 0x53, 0x63, 0x54, 0x67, 0x55, 0x71,
    0x56, 0x57, 0x57, 0x59, 0x58, 0x03, 0x59, 0x00, 0x5a, 0x4a,
    0xa1, 0x00, 0xa2, 0x00, 0xa3, 0x00, 0xa4, 0x00, 0xa8, 0x00, 0xa9, 0x00,
    0xaa, 0xdf, 0xab, 0x7f, 0xac, 0x00, 0xad, 0x41, 0xae, 0x00, 0xaf, 0x00,
    0xa0, 0x44
    },
    { /* DUAL_1024_768_8 */
    0x30, 0x3a, 0x31, 0x83, 0x32, 0x38, 0x33, 0x00, 0x34, 0x40, 0x3e, 0x00,
    0x3f, 0x00, 0x40, 0x80, 0x41, 0x00, 0x42, 0x00, 0x43, 0x00, 0x44, 0x00,
    0x45, 0x42, 0x46, 0x00, 0x47, 0xfc, 0x48, 0x20, 0x49, 0x1c, 0x4a, 0x41,
    0x4b, 0xa0, 0x4c, 0x00,
    0x50, 0x06, 0x51, 0x68, 0x52, 0xa7, 0x53, 0x7f, 0x54, 0x83, 0x55, 0x25,
    0x56, 0xff, 0x57, 0x03, 0x58, 0x04, 0x59, 0x00, 0x5a, 0xc2,
    0xa1, 0x00, 0xa2, 0x00, 0xa3, 0x00, 0xa4, 0x00, 0xa8, 0x00, 0xa9, 0x00,
    0xaa, 0xdf, 0xab, 0x7f, 0xac, 0x00, 0xad, 0x41, 0xae, 0x00, 0xaf, 0x00,
    0xa0, 0x44
    },
    { /* SINGLE_800_600_8 */
    0x30, 0x36, 0x31, 0x82, 0x32, 0x38, 0x33, 0x00, 0x34, 0x40, 0x3e, 0x03,
    0x3f, 0xff, 0x40, 0x64, 0x41, 0x00, 0x42, 0x00, 0x43, 0x00, 0x44, 0xc8,
    0x45, 0x02, 0x46, 0x00, 0x47, 0xfc, 0x48, 0x20, 0x49, 0x1c, 0x4a, 0x00,
    0x4b, 0xa0, 0x4c, 0x00,
    0x50, 0x04, 0x51, 0x48, 0x52, 0x83, 0x53, 0x63, 0x54, 0x67, 0x55, 0x71,
    0x56, 0x57, 0x57, 0x59, 0x58, 0x03, 0x59, 0x00, 0x5a, 0x4a,
    0xa1, 0x00, 0xa2, 0x00, 0xa3, 0x00, 0xa4, 0x00, 0xa8, 0x00, 0xa9, 0x00,
    0xaa, 0xdf, 0xab, 0x7f, 0xac, 0x00, 0xad, 0x41, 0xae, 0x00, 0xaf, 0x00,
    0xa0, 0x44
    },
    { /* SINGLE_1024_768_8 */
    0x30, 0x3a, 0x31, 0x82, 0x32, 0x38, 0x33, 0x00, 0x34, 0x40, 0x3e, 0x00,
    0x3f, 0x00, 0x40, 0x80, 0x41, 0x00, 0x42, 0x00, 0x43, 0x00, 0x44, 0x00,
    0x45, 0x42, 0x46, 0x00, 0x47, 0xfc, 0x48, 0x20, 0x49, 0x1c, 0x4a, 0x11,
    0x4b, 0xa0, 0x4c, 0x00,
    0x50, 0x06, 0x51, 0x68, 0x52, 0xa7, 0x53, 0x7f, 0x54, 0x83, 0x55, 0x24,
    0x56, 0xff, 0x57, 0x03, 0x58, 0x04, 0x59, 0x00, 0x5a, 0xc2,
    0xa1, 0x00, 0xa2, 0x00, 0xa3, 0x00, 0xa4, 0x00, 0xa8, 0x00, 0xa9, 0x00,
    0xaa, 0xdf, 0xab, 0x7f, 0xac, 0x00, 0xad, 0x41, 0xae, 0x00, 0xaf, 0x00,
    0xa0, 0x44
    },
    { /* SINGLE_1280_1024_8 */
    0x30, 0x3e, 0x31, 0x82, 0x32, 0x38, 0x33, 0x00, 0x34, 0x40, 0x3e, 0x00,
    0x3f, 0x00, 0x40, 0xa0, 0x41, 0x00, 0x42, 0x00, 0x43, 0x00, 0x44, 0x00,
    0x45, 0x42, 0x46, 0x00, 0x47, 0xfc, 0x48, 0x20, 0x49, 0x1c, 0x4a, 0x11,
    0x4b, 0xa0, 0x4c, 0x00,
    0x50, 0x08, 0x51, 0x88, 0x52, 0xd3, 0x53, 0x9f, 0x54, 0xa3, 0x55, 0x2a,
    0x56, 0xff, 0x57, 0x04, 0x58, 0x00, 0x59, 0x00, 0x5a, 0x63,
    0xa1, 0x00, 0xa2, 0x00, 0xa3, 0x00, 0xa4, 0x00, 0xa8, 0x00, 0xa9, 0x00,
    0xaa, 0xdf, 0xab, 0x7f, 0xac, 0x00, 0xad, 0x41, 0xae, 0x00, 0xaf, 0x00,
    0xa0, 0x44
    },
    { /* TV_MODE_CCIR */
    0x30, 0x24, 0x31, 0x84, 0x32, 0x20, 0x33, 0x09, 0x34, 0xf0, 0x3e, 0x03,
    0x3f, 0xff, 0x40, 0x64, 0x41, 0x00, 0x42, 0x00, 0x43, 0x00, 0x44, 0xc8,
    0x45, 0x02, 0x46, 0x00, 0x47, 0xfc, 0x48, 0x20, 0x49, 0x1c, 0x4a, 0x00,
    0x4b, 0xa0, 0x4c, 0x00,
    0x50, 0x04, 0x51, 0x48, 0x52, 0x83, 0x53, 0x63, 0x54, 0x68, 0x55, 0x73,
    0x56, 0x57, 0x57, 0x58, 0x58, 0x04, 0x59, 0x57, 0x5a, 0x7b,
    0xa1, 0x10, 0xa2, 0xab, 0xa3, 0x98, 0xa4, 0xc1, 0xa8, 0x8c, 0xa9, 0x05,
    0xaa, 0x17, 0xab, 0x35, 0xac, 0x41, 0xad, 0x68, 0xae, 0x00, 0xaf, 0x00,
    0xa0, 0x00
    },
    { /* TV_MODE_EIA */
    0x30, 0x24, 0x31, 0x84, 0x32, 0x20, 0x33, 0x09, 0x34, 0xf0, 0x3e, 0x03,
    0x3f, 0xff, 0x40, 0x64, 0x41, 0x00, 0x42, 0x00, 0x43, 0x00, 0x44, 0xc8,
    0x45, 0x02, 0x46, 0x00, 0x47, 0xfc, 0x48, 0x20, 0x49, 0x1c, 0x4a, 0x00,
    0x4b, 0xa0, 0x4c, 0x00,
    0x50, 0x04, 0x51, 0x48, 0x52, 0x83, 0x53, 0x63, 0x54, 0x68, 0x55, 0x73,
    0x56, 0x57, 0x57, 0x58, 0x58, 0x04, 0x59, 0x57, 0x5a, 0x7b,
    0xa1, 0x10, 0xa2, 0xab, 0xa3, 0x98, 0xa4, 0xc1, 0xa8, 0x8c, 0xa9, 0x05,
    0xaa, 0x17, 0xab, 0x35, 0xac, 0x41, 0xad, 0x68, 0xae, 0x00, 0xaf, 0x00,
    0xa0, 0x00
    },
};

static char SMI_GCR[18] = {
    0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x40,
    0x06, 0x05, 0x07, 0x0f, 0x08, 0xff
};

/*******************************************************************************
*
* Read SMI ISA register
*/
static char smiRead (unsigned short index, char reg)
{
    register GraphicDevice *pGD = (GraphicDevice *)&smi;

    out8 ((pGD->isaBase + index), reg);
    return (in8 (pGD->isaBase + index + 1));
}

/*******************************************************************************
*
* Write SMI ISA register
*/
static void smiWrite (unsigned short index, char reg, char val)
{
    register GraphicDevice *pGD = (GraphicDevice *)&smi;

    out8 ((pGD->isaBase + index), reg);
    out8 ((pGD->isaBase + index + 1), val);
}

/*******************************************************************************
*
* Write a table of SMI ISA register
*/
static void smiLoadRegs (
    unsigned int iReg,
    unsigned int dReg,
    char         *regTab,
    unsigned int tabSize
    )
{
    register int i;

    for (i=0; i<tabSize; i+=2)
    {
        out8 (iReg, regTab[i]);
        out8 (dReg, regTab[i+1]);
    }
}

/*******************************************************************************
*
* Init capture port registers
*/
static void smiInitCapturePort (void)
{
    SmiCapturePort smiCP = { 0x01400600, 0x30, 0x40, 480, 640, 0, 0, 2560, 6 };
    register GraphicDevice *pGD  = (GraphicDevice *)&smi;
    register SmiCapturePort *pCP = (SmiCapturePort *)&smiCP;

    out32r ((pGD->cprBase + 0x0004), ((pCP->topClip<<16)   | pCP->leftClip));
    out32r ((pGD->cprBase + 0x0008), ((pCP->srcHeight<<16) | pCP->srcWidth));
    out32r ((pGD->cprBase + 0x000c), pCP->srcBufStart1/8);
    out32r ((pGD->cprBase + 0x0010), pCP->srcBufStart2/8);
    out32r ((pGD->cprBase + 0x0014), pCP->srcOffset/8);
    out32r ((pGD->cprBase + 0x0018), pCP->fifoControl);
    out32r ((pGD->cprBase + 0x0000), pCP->control);
}


/*******************************************************************************
*
* Init video processor registers
*/
static void smiInitVideoProcessor (void)
{
    SmiVideoProc smiVP = { 0x100000, 0, 0, 0, 0, 1600, 0x1200543, 4, 0xededed };
    SmiVideoWin  smiVW = { 0, 0, 599, 799, 0, 1600, 0, 0, 0 };
    register GraphicDevice *pGD = (GraphicDevice *)&smi;
    register SmiVideoProc  *pVP = (SmiVideoProc *)&smiVP;
    register SmiVideoWin *pVWin = (SmiVideoWin *)&smiVW;

    pVP->width    = pGD->plnSizeX * pGD->gdfBytesPP;
    pVP->control |= pGD->gdfIndex << 16;
    pVWin->bottom = pGD->winSizeY - 1;
    pVWin->right  = pGD->winSizeX - 1;
    pVWin->width  = pVP->width;

    /* color key */
    out32r ((pGD->vprBase + 0x0004), pVP->colorKey);

     /* color key mask */
    out32r ((pGD->vprBase + 0x0008), pVP->colorKeyMask);

    /* data src start adrs */
    out32r ((pGD->vprBase + 0x000c), pVP->start / 8);

    /* data width and offset */
    out32r ((pGD->vprBase + 0x0010),
        ((pVP->offset   / 8 * pGD->gdfBytesPP) << 16) |
         (pGD->plnSizeX / 8 * pGD->gdfBytesPP));

    /* video window 1 */
    out32r ((pGD->vprBase + 0x0014),
        ((pVWin->top << 16) | pVWin->left));

    out32r ((pGD->vprBase + 0x0018),
        ((pVWin->bottom << 16) | pVWin->right));

    out32r ((pGD->vprBase + 0x001c), pVWin->srcStart / 8);

    out32r ((pGD->vprBase + 0x0020),
        (((pVWin->offset / 8) << 16) | (pVWin->width / 8)));

    out32r ((pGD->vprBase + 0x0024),
        (((pVWin->hStretch) << 8) | pVWin->vStretch));

    /* video window 2 */
    out32r ((pGD->vprBase + 0x0028),
        ((pVWin->top << 16) | pVWin->left));

    out32r ((pGD->vprBase + 0x002c),
        ((pVWin->bottom << 16) | pVWin->right));

    out32r ((pGD->vprBase + 0x0030),
        pVWin->srcStart / 8);

    out32r ((pGD->vprBase + 0x0034),
        (((pVWin->offset / 8) << 16) | (pVWin->width / 8)));

    out32r ((pGD->vprBase + 0x0038),
        (((pVWin->hStretch) << 8) | pVWin->vStretch));

    /* fifo prio control */
    out32r ((pGD->vprBase + 0x0054), pVP->fifoPrio);

    /* fifo empty request levell */
    out32r ((pGD->vprBase + 0x0058), pVP->fifoERL);

    /* conversion constant */
    out32r ((pGD->vprBase + 0x005c), pVP->YUVtoRGB);

    /* vpr control word */
    out32r ((pGD->vprBase + 0x0000), pVP->control);
}

/******************************************************************************
 *
 * Init drawing engine registers
 */
static void smiInitDrawingEngine (void)
{
    GraphicDevice *pGD = (GraphicDevice *)&smi;
    unsigned int val;

    /* don't start now */
    out32r ((pGD->dprBase + 0x000c), 0x000f0000);

    /* set rop2 to copypen */
    val = 0xffff3ff0 & in32r ((pGD->dprBase + 0x000c));
    out32r ((pGD->dprBase + 0x000c), (val | 0x8000 | 0x0c));

    /* set clip rect */
    out32r ((pGD->dprBase + 0x002c), 0);
    out32r ((pGD->dprBase + 0x0030),
        ((pGD->winSizeY<<16) | pGD->winSizeX * pGD->gdfBytesPP ));

    /* src row pitch */
    val = 0xffff0000 & (in32r ((pGD->dprBase + 0x0010)));
    out32r ((pGD->dprBase + 0x0010),
        (val | pGD->plnSizeX * pGD->gdfBytesPP));

    /* dst row pitch */
    val = 0x0000ffff & (in32r ((pGD->dprBase + 0x0010)));
    out32r ((pGD->dprBase + 0x0010),
        (((pGD->plnSizeX * pGD->gdfBytesPP)<<16) | val));

    /* window width src/dst */
    out32r ((pGD->dprBase + 0x003c),
        (((pGD->plnSizeX * pGD->gdfBytesPP & 0x0fff)<<16) |
          (pGD->plnSizeX * pGD->gdfBytesPP & 0x0fff)));
    out16r ((pGD->dprBase + 0x001e), 0x0000);

    /* src base adrs */
    out32r ((pGD->dprBase + 0x0040),
        (((pGD->frameAdrs/8) & 0x000fffff)));

    /* dst base adrs */
    out32r ((pGD->dprBase + 0x0044),
        (((pGD->frameAdrs/8) & 0x000fffff)));

    /* foreground color */
    out32r ((pGD->dprBase + 0x0014), pGD->fg);

    /* background color */
    out32r ((pGD->dprBase + 0x0018), pGD->bg);

    /* xcolor */
    out32r ((pGD->dprBase + 0x0020), 0x00ffffff);

    /* xcolor mask */
    out32r ((pGD->dprBase + 0x0024), 0x00ffffff);

    /* bit mask */
    out32r ((pGD->dprBase + 0x0028), 0x00ffffff);

    /* load mono pattern */
    out32r ((pGD->dprBase + 0x0034), 0);
    out32r ((pGD->dprBase + 0x0038), 0);
}

static struct pci_device_id supported[] = {
    { PCI_VENDOR_ID_SMI, PCI_DEVICE_ID_SMI_710 },
    { PCI_VENDOR_ID_SMI, PCI_DEVICE_ID_SMI_712 },
    { PCI_VENDOR_ID_SMI, PCI_DEVICE_ID_SMI_810 },
    { }
};


/*******************************************************************************
*
* Init video chip with common Linux graphic modes (lilo)
*/
void *video_hw_init (void)
{
    GraphicDevice *pGD = (GraphicDevice *)&smi;
    unsigned short device_id;
    pci_dev_t devbusfn;
    int videomode;
    unsigned int pci_mem_base, *vm, i;
    unsigned int gdfTab[] = { 1, 2, 2, 4, 3, 1 };
    char *penv;
    char *gdfModes[] =
            {
            "8 Bit Index Color",
            "15 Bit 5-5-5 RGB",
            "16 Bit 5-6-5 RGB",
            "32 Bit X-8-8-8 RGB",
            "24 Bit 8-8-8 RGB",
            "8 Bit 3-3-2 RGB"
            };
    int vgaModes[16][2] =
            {
            {769, -1}, {771, 0x00002}, {773, 0x00003}, {775, 0x00004},
            {784, -1}, {787, 0x10002}, {790, 0x10003}, {793, 0x10004},
            {785, -1}, {788, 0x20002}, {791, 0x20003}, {794, 0x20004},
            {786, -1}, {789, 0x40002}, {792, 0x40003}, {795, 0x40004}
            };

    /* Search for video chip */
    printf("Video: ");

    if ((devbusfn = pci_find_devices(supported, 0)) < 0)
    {
        printf ("Controller not found !\n");
        return (NULL);
    }

    /* PCI setup */
    pci_write_config_dword (devbusfn, PCI_COMMAND, (PCI_COMMAND_MEMORY | PCI_COMMAND_IO));
    pci_read_config_word (devbusfn, PCI_DEVICE_ID, &device_id);
    pci_read_config_dword (devbusfn, PCI_BASE_ADDRESS_0, &pci_mem_base);
    pci_mem_base = pci_mem_to_phys (devbusfn, pci_mem_base);

    /* Initialize the video controller */
    if ((penv = getenv ("videomode")) != NULL)
        videomode = (int)simple_strtoul (penv, NULL, 16);
    else
        videomode = 0x303;                    /* Default 800x600 8 bit index color */

    /* Compare with common vga mode numbers */
    for (i=0; i<16; i++)
    {
        if (vgaModes[i][0] == videomode)
        {
            if (vgaModes[i][1] == -1)
            {
                printf("Videomode not supported !\n");
                return (NULL);                /* mode not supported */
            }
            pGD->mode = vgaModes[i][1];        /* use driver int. mode number */
            break;
        }
    }

    /* Extract graphic data format */
    pGD->gdfIndex = (pGD->mode & 0x00070000) >> 16;
    if (pGD->gdfIndex > 5)
        pGD->gdfIndex = 0;
    pGD->gdfBytesPP = gdfTab[pGD->gdfIndex];

    /* Extract graphic resolution */
    pGD->mode &= 0xf;

    /* Exit for not supported resolutions */
    if (((pGD->mode==DUAL_800_600) || (pGD->mode==DUAL_1024_768)) && (pGD->gdfBytesPP > 1))
    {
        printf ("Dual screen for 1BPP only !\n");
        return (NULL);
    }

    if ((pGD->mode==SINGLE_1280_1024) && (pGD->gdfBytesPP==4))
    {
        printf ("Out of memory !\n");
        return (NULL);
    }

    /* Set graphic parameters */
    switch (pGD->mode)
    {
    case DUAL_800_600:
            pGD->winSizeX = 800;
            pGD->winSizeY = 600;
            pGD->plnSizeX = 1600;
            pGD->plnSizeY = 600;
            sprintf (pGD->modeIdent, "Dual Screen 800x600 with %s", gdfModes[pGD->gdfIndex]);
            break;
    case DUAL_1024_768:
            pGD->winSizeX = 1024;
            pGD->winSizeY = 768;
            pGD->plnSizeX = 2048;
            pGD->plnSizeY = 768;
            sprintf (pGD->modeIdent, "Dual Screen 1024x768 with %s", gdfModes[pGD->gdfIndex]);
            break;
    case SINGLE_800_600:
            pGD->winSizeX = 800;
            pGD->winSizeY = 600;
            pGD->plnSizeX = 800;
            pGD->plnSizeY = 600;
            sprintf (pGD->modeIdent, "Single Screen 800x600 with %s", gdfModes[pGD->gdfIndex]);
            break;
    case SINGLE_1024_768:
            pGD->winSizeX = 1024;
            pGD->winSizeY = 768;
            pGD->plnSizeX = 1024;
            pGD->plnSizeY = 768;
            sprintf (pGD->modeIdent,"Single Screen 1024x768 with %s", gdfModes[pGD->gdfIndex]);
            break;
    case TV_MODE_CCIR:
            pGD->winSizeX = 720;
            pGD->winSizeY = 576;
            pGD->plnSizeX = 720;
            pGD->plnSizeY = 576;
            sprintf (pGD->modeIdent, "TV Mode CCIR with %s", gdfModes[pGD->gdfIndex]);
            break;
    case TV_MODE_EIA:
            pGD->winSizeX = 720;
            pGD->winSizeY = 484;
            pGD->plnSizeX = 720;
            pGD->plnSizeY = 484;
            sprintf (pGD->modeIdent, "TV Mode EIA with %s", gdfModes[pGD->gdfIndex]);
            break;
    case SINGLE_1280_1024:
            pGD->winSizeX = 1280;
            pGD->winSizeY = 1024;
            pGD->plnSizeX = 1280;
            pGD->plnSizeY = 1024;
            sprintf (pGD->modeIdent, "Single Screen 1280x1024 with %s", gdfModes[pGD->gdfIndex]);
            break;
    default:
            printf("Videomode not supported !\n");
            return (NULL);
    }


    pGD->isaBase = CFG_ISA_IO;
    pGD->pciBase = pci_mem_base;
    pGD->dprBase = (pci_mem_base + 0x400000 + 0x8000);
    pGD->vprBase = (pci_mem_base + 0x400000 + 0xc000);
    pGD->cprBase = (pci_mem_base + 0x400000 + 0xe000);
    pGD->frameAdrs = pci_mem_base;
    pGD->memSize = VIDEO_MEM_SIZE;

    /* Turn off display */
    smiWrite (0x3c4, 0x01, 0x20);

    /* Unlock ext. crt regs */
    out8 (SMI_LOCK_REG, 0x40);

    /* Set Register base to isa 3dx for 3?x regs (color mode) */
    out8 (SMI_MISC_REG, 0x2b);

    /* Unlock crt regs 0-7 */
    smiWrite (0x3d4, 0x11, 0x0e);

    /* Sytem Control Register */
    smiLoadRegs (SMI_INDX_C4, SMI_DATA_C5,
         SMI_SCR, sizeof(SMI_SCR));

    /* extented CRT Register */
    smiLoadRegs (SMI_INDX_D4, SMI_DATA_D5,
         SMI_EXT_CRT[pGD->mode], sizeof(SMI_EXT_CRT)/VIDEO_MODES);

    /* Sequencer Register */
    smiLoadRegs (SMI_INDX_C4, SMI_DATA_C5,
         SMI_SEQR, sizeof(SMI_SEQR));

    /* Power Control Register */
    smiLoadRegs (SMI_INDX_C4, SMI_DATA_C5,
         SMI_PCR[pGD->mode], sizeof(SMI_PCR)/VIDEO_MODES);

    /* Memory Control Register */
    smiLoadRegs (SMI_INDX_C4, SMI_DATA_C5,
         SMI_MCR[pGD->mode], sizeof(SMI_MCR)/VIDEO_MODES);

    /* Clock Control Register */
    smiLoadRegs (SMI_INDX_C4, SMI_DATA_C5,
         SMI_CCR[pGD->mode], sizeof(SMI_CCR)/VIDEO_MODES);

    /* Shadow VGA Register */
    smiLoadRegs (SMI_INDX_D4, SMI_DATA_D5,
         SMI_SHVGA[pGD->mode], sizeof(SMI_SHVGA)/VIDEO_MODES);

    /* General Purpose Register */
    smiLoadRegs (SMI_INDX_C4, SMI_DATA_C5,
         SMI_GPR[pGD->mode], sizeof(SMI_GPR)/VIDEO_MODES);

    /* Hardware Cusor Register */
    smiLoadRegs (SMI_INDX_C4, SMI_DATA_C5,
         SMI_HCR[pGD->mode], sizeof(SMI_HCR)/VIDEO_MODES);

    /* Flat Panel Register */
    smiLoadRegs (SMI_INDX_C4, SMI_DATA_C5,
         SMI_FPR[pGD->mode], sizeof(SMI_FPR)/VIDEO_MODES);

    /* CRTC Register */
    smiLoadRegs (SMI_INDX_D4, SMI_DATA_D5,
         SMI_CRTCR[pGD->mode], sizeof(SMI_CRTCR)/VIDEO_MODES);

    /* Graphics Controller Register */
    smiLoadRegs (SMI_INDX_CE, SMI_DATA_CF,
         SMI_GCR, sizeof(SMI_GCR));

    /* Patch memory and refresh settings for SMI710 */
    if (device_id == PCI_DEVICE_ID_SMI_710)
    {
    unsigned char reg = smiRead (0x3c4, 0x62);

    /* external memory disabled */
    smiWrite (0x3c4, 0x62, (reg & 0xfb));
    /* memory clock */
    smiWrite (0x3c4, 0x6a, 0x75);
    }

    /* Patch memory and refresh settings for SMI712 */
    if (device_id == PCI_DEVICE_ID_SMI_712)
    {
    unsigned char reg = smiRead (0x3c4, 0x62);

    /* IL runs at MCLK; 64bit bus; external memory disabled */
    smiWrite (0x3c4, 0x62, (reg | 0xc4));
    /* memory clock */
    smiWrite (0x3c4, 0x6a, 0x80);
    }

    /* Patch clock settings for SMI810 */
    if (device_id == PCI_DEVICE_ID_SMI_810)
    {
    /* clock control */
    smiWrite (0x3c4, 0x69, 0x03);
    }

    /* Video processor default setup */
    smiInitVideoProcessor ();

    /* Capture port default setup */
    smiInitCapturePort ();

    /* Drawing engine default setup */
    smiInitDrawingEngine ();

    /* Turn on display */
    smiWrite (0x3c4, 0x01, 0x01);

    /* Clear video memory */
    i = pGD->memSize/4;
    vm = (unsigned int *)pGD->pciBase;
    while(i--)
    *vm++ = 0;

    printf("mode=%x - %s\n", videomode, pGD->modeIdent);
    return ((void*)&smi);
}

/*******************************************************************************
*
* Drawing engine fill on screen region
*/
void video_hw_rectfill (
    unsigned int bpp,             /* bytes per pixel */
    unsigned int dst_x,           /* dest pos x */
    unsigned int dst_y,           /* dest pos y */
    unsigned int dim_x,           /* frame width */
    unsigned int dim_y,           /* frame height */
    unsigned int color            /* fill color */
     )
{
    register GraphicDevice *pGD = (GraphicDevice *)&smi;
    register unsigned int control;

    dim_x *= bpp;

    out32r ((pGD->dprBase + 0x0014), color);
    out32r ((pGD->dprBase + 0x0004), ((dst_x<<16) | dst_y));
    out32r ((pGD->dprBase + 0x0008), ((dim_x<<16) | dim_y));

    control = 0x0000ffff &  in32r ((pGD->dprBase + 0x000c));

    control |= 0x80010000;

    out32r ((pGD->dprBase + 0x000c),  control);

    /* Wait for drawing processor */
    do
    {
    out8 ((pGD->isaBase + 0x3c4), 0x16);
    } while (in8 (pGD->isaBase + 0x3c5) & 0x08);
}

/*******************************************************************************
*
* Drawing engine bitblt with screen region
*/
void video_hw_bitblt (
    unsigned int bpp,             /* bytes per pixel */
    unsigned int src_x,           /* source pos x */
    unsigned int src_y,           /* source pos y */
    unsigned int dst_x,           /* dest pos x */
    unsigned int dst_y,           /* dest pos y */
    unsigned int dim_x,           /* frame width */
    unsigned int dim_y            /* frame height */
    )
{
    register GraphicDevice *pGD = (GraphicDevice *)&smi;
    register unsigned int control;

    dim_x *= bpp;

    if ((src_y<dst_y) || ((src_y==dst_y) && (src_x<dst_x)))
    {
    out32r ((pGD->dprBase + 0x0000), (((src_x+dim_x-1)<<16) | (src_y+dim_y-1)));
    out32r ((pGD->dprBase + 0x0004), (((dst_x+dim_x-1)<<16) | (dst_y+dim_y-1)));
    control = 0x88000000;
    }
    else
    {
    out32r ((pGD->dprBase + 0x0000), ((src_x<<16) | src_y));
    out32r ((pGD->dprBase + 0x0004), ((dst_x<<16) | dst_y));
    control = 0x80000000;
    }

    out32r ((pGD->dprBase + 0x0008), ((dim_x<<16) | dim_y));
    control |= (0x0000ffff &  in32r ((pGD->dprBase + 0x000c)));
    out32r ((pGD->dprBase + 0x000c), control);

    /* Wait for drawing processor */
    do
    {
    out8 ((pGD->isaBase + 0x3c4), 0x16);
    } while (in8 (pGD->isaBase + 0x3c5) & 0x08);
}

/*******************************************************************************
*
* Set a RGB color in the LUT (8 bit index)
*/
void video_set_lut (
    unsigned int index,           /* color number */
    unsigned char r,              /* red */
    unsigned char g,              /* green */
    unsigned char b               /* blue */
    )
{
    register GraphicDevice *pGD = (GraphicDevice *)&smi;

    out8 (SMI_LUT_MASK,  0xff);

    out8 (SMI_LUT_START, (char)index);

    out8 (SMI_LUT_RGB, r>>2);    /* red */
    udelay (10);
    out8 (SMI_LUT_RGB, g>>2);    /* green */
    udelay (10);
    out8 (SMI_LUT_RGB, b>>2);    /* blue */
    udelay (10);
}

#endif /* CONFIG_VIDEO_SMI_LYNXEM */
