/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

const { Actor } = require("resource://devtools/shared/protocol.js");
const {
  webExtensionInspectedWindowSpec,
} = require("resource://devtools/shared/specs/addon/webextension-inspected-window.js");

ChromeUtils.defineESModuleGetters(
  this,
  {
    ExtensionUtils: "resource://gre/modules/ExtensionUtils.sys.mjs",
  },
  { global: "contextual" }
);

const {
  DevToolsServer,
} = require("resource://devtools/server/devtools-server.js");

loader.lazyGetter(
  this,
  "NodeActor",
  () =>
    require("resource://devtools/server/actors/inspector/node.js").NodeActor,
  true
);

// A weak set of the documents for which a warning message has been
// already logged (so that we don't keep emitting the same warning if an
// extension keeps calling the devtools.inspectedWindow.eval API method
// when it fails to retrieve a result, but we do log the warning message
// if the user reloads the window):
//
// WeakSet<Document>
const deniedWarningDocuments = new WeakSet();

function isSystemPrincipalWindow(window) {
  return window.document.nodePrincipal.isSystemPrincipal;
}

// Create the exceptionInfo property in the format expected by a
// WebExtension inspectedWindow.eval API calls.
function createExceptionInfoResult(props) {
  return {
    exceptionInfo: {
      isError: true,
      code: "E_PROTOCOLERROR",
      description: "Unknown Inspector protocol error",

      // Apply the passed properties.
      ...props,
    },
  };
}

// Show a warning message in the webconsole when an extension
// eval request has been denied, so that the user knows about it
// even if the extension doesn't report the error itself.
function logAccessDeniedWarning(window, callerInfo, extensionPolicy) {
  // Do not log the same warning multiple times for the same document.
  if (deniedWarningDocuments.has(window.document)) {
    return;
  }

  deniedWarningDocuments.add(window.document);

  const { name } = extensionPolicy;

  // System principals have a null nodePrincipal.URI and so we use
  // the url from window.location.href.
  const reportedURIorPrincipal = isSystemPrincipalWindow(window)
    ? Services.io.newURI(window.location.href)
    : window.document.nodePrincipal;

  const error = Cc["@mozilla.org/scripterror;1"].createInstance(
    Ci.nsIScriptError
  );

  const msg = `The extension "${name}" is not allowed to access ${reportedURIorPrincipal.spec}`;

  const innerWindowId = window.windowGlobalChild.innerWindowId;

  const errorFlag = 0;

  let { url, lineNumber } = callerInfo;

  const callerURI = callerInfo.url && Services.io.newURI(callerInfo.url);

  // callerInfo.url is not the full path to the file that called the WebExtensions
  // API yet (Bug 1448878), and so we associate the error to the url of the extension
  // manifest.json file as a fallback.
  if (callerURI.filePath === "/") {
    url = extensionPolicy.getURL("/manifest.json");
    lineNumber = null;
  }

  error.initWithWindowID(
    msg,
    url,
    lineNumber,
    0,
    0,
    errorFlag,
    "webExtensions",
    innerWindowId
  );
  Services.console.logMessage(error);
}

/**
 * @param {WebExtensionPolicy} extensionPolicy
 * @param {nsIPrincipal} principal
 * @param {Location} location
 * @returns {boolean} Whether the extension is allowed to run code in execution
 *   contexts with the given principal.
 */
function extensionAllowedToInspectPrincipal(
  extensionPolicy,
  principal,
  location
) {
  if (principal.isNullPrincipal) {
    if (location.protocol === "view-source:") {
      // Don't fall back to the precursor, we never want extensions to be able
      // to run code in view-source:-documents.
      return false;
    }
    // data: and sandboxed documents.
    //
    // Rather than returning true unconditionally, we go through additional
    // checks to prevent execution in sandboxed documents created by principals
    // that extensions cannot access otherwise.
    principal = principal.precursorPrincipal;
    if (!principal) {
      // Top-level about:blank, etc.
      return true;
    }
  }
  if (!principal.isContentPrincipal) {
    return false;
  }
  const principalURI = principal.URI;
  if (principalURI.schemeIs("https") || principalURI.schemeIs("http")) {
    if (WebExtensionPolicy.isRestrictedURI(principalURI)) {
      return false;
    }
    if (extensionPolicy.quarantinedFromURI(principalURI)) {
      return false;
    }
    // Common case: http(s) allowed.
    return true;
  }

  if (ExtensionUtils.isExtensionUrl(principalURI)) {
    // Ordinarily, we don't allow extensions to execute arbitrary code in
    // their own context. The devtools.inspectedWindow.eval API is a special
    // case - this can only be used through the devtools_page feature, which
    // requires the user to open the developer tools first. If an extension
    // really wants to debug itself, we let it do so.
    return extensionPolicy.id === principal.addonId;
  }

  if (principalURI.schemeIs("file")) {
    return true;
  }

  return false;
}

class CustomizedReload {
  constructor(params) {
    this.docShell = params.targetActor.window.docShell;
    this.docShell.QueryInterface(Ci.nsIWebProgress);

    this.inspectedWindowEval = params.inspectedWindowEval;
    this.callerInfo = params.callerInfo;

    this.ignoreCache = params.ignoreCache;
    this.injectedScript = params.injectedScript;

    this.customizedReloadWindows = new WeakSet();
  }

  QueryInterface = ChromeUtils.generateQI([
    "nsIWebProgressListener",
    "nsISupportsWeakReference",
  ]);

  get window() {
    return this.docShell.DOMWindow;
  }

  get webNavigation() {
    return this.docShell
      .QueryInterface(Ci.nsIInterfaceRequestor)
      .getInterface(Ci.nsIWebNavigation);
  }

  get browsingContext() {
    return this.docShell.browsingContext;
  }

  start() {
    if (!this.waitForReloadCompleted) {
      this.waitForReloadCompleted = new Promise((resolve, reject) => {
        this.resolveReloadCompleted = resolve;
        this.rejectReloadCompleted = reject;

        let reloadFlags = Ci.nsIWebNavigation.LOAD_FLAGS_NONE;

        if (this.ignoreCache) {
          reloadFlags |= Ci.nsIWebNavigation.LOAD_FLAGS_BYPASS_CACHE;
        }

        try {
          if (this.injectedScript) {
            // Listen to the newly created document elements only if there is an
            // injectedScript to evaluate.
            Services.obs.addObserver(this, "initial-document-element-inserted");
          }

          // Watch the loading progress and clear the current CustomizedReload once the
          // page has been reloaded (or if its reloading has been interrupted).
          this.docShell.addProgressListener(
            this,
            Ci.nsIWebProgress.NOTIFY_STATE_DOCUMENT
          );

          this.webNavigation.reload(reloadFlags);
        } catch (err) {
          // Cancel the injected script listener if the reload fails
          // (which will also report the error by rejecting the promise).
          this.stop(err);
        }
      });
    }

    return this.waitForReloadCompleted;
  }

  observe(subject, topic) {
    if (topic !== "initial-document-element-inserted") {
      return;
    }

    const document = subject;
    const window = document?.defaultView;

    // Filter out non interesting documents.
    if (!document || !document.location || !window) {
      return;
    }

    const subjectDocShell = window.docShell;

    // Keep track of the set of window objects where we are going to inject
    // the injectedScript: the top level window and all its descendant
    // that are still of type content (filtering out loaded XUL pages, if any).
    if (window == this.window) {
      this.customizedReloadWindows.add(window);
    } else if (subjectDocShell.sameTypeParent) {
      const parentWindow = subjectDocShell.sameTypeParent.domWindow;
      if (parentWindow && this.customizedReloadWindows.has(parentWindow)) {
        this.customizedReloadWindows.add(window);
      }
    }

    if (t                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                