/*
 * Copyright (C) 2021-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "ArgumentCoders.h"
#include "BackgroundFetchState.h"
#include "Connection.h"
#include "ConnectionHandle.h"
#include "DataTaskIdentifier.h"
#include "DownloadID.h"
#include "ITPThirdPartyData.h"
#include "MessageNames.h"
#include "NavigatingToAppBoundDomain.h"
#include "NetworkProcessCreationParameters.h"
#include "QuotaIncreaseRequestIdentifier.h"
#include "SandboxExtension.h"
#include "WebPageProxyIdentifier.h"
#include "WebPushMessage.h"
#include "WebsiteDataFetchOption.h"
#include "WebsiteDataStoreParameters.h"
#include "WebsiteDataType.h"
#include <WebCore/CrossSiteNavigationDataTransfer.h>
#include <WebCore/NotificationPayload.h>
#include <WebCore/OrganizationStorageAccessPromptQuirk.h>
#include <WebCore/PageIdentifier.h>
#include <WebCore/ProcessIdentifier.h>
#include <WebCore/RegistrableDomain.h>
#include <WebCore/SecurityOriginData.h>
#include <optional>
#include <span>
#include <utility>
#include <wtf/Forward.h>
#include <wtf/HashSet.h>
#include <wtf/OptionSet.h>
#include <wtf/ThreadSafeRefCounted.h>
#include <wtf/UUID.h>
#include <wtf/Vector.h>
#include <wtf/WallTime.h>
#include <wtf/text/WTFString.h>

namespace IPC {
class FormDataReference;
}

namespace PAL {
class SessionID;
}

namespace WebCore {
class CertificateInfo;
class CurlProxySettings;
class PrivateClickMeasurement;
class RegistrableDomain;
class ResourceRequest;
class SecurityOriginData;
enum class NotificationEventType : bool;
enum class SameSiteStrictEnforcementEnabled : bool;
enum class FirstPartyWebsiteDataRemovalMode : uint8_t;
enum class HTTPCookieAcceptPolicy : uint8_t;
enum class PushPermissionState : uint8_t;
enum class StoredCredentialsPolicy : uint8_t;
enum class ThirdPartyCookieBlockingMode : uint8_t;
struct ClientOrigin;
struct NotificationData;
struct SoupNetworkProxySettings;
}

namespace WebKit {
class WebPageNetworkParameters;
enum class CallDownloadDidStart : bool;
enum class DidFilterKnownLinkDecoration : bool;
enum class LoadedWebArchive : bool;
enum class ShouldGrandfatherStatistics : bool;
enum class CacheModel : uint8_t;
enum class RemoteWorkerType : uint8_t;
struct AppPrivacyReportTestingData;
struct NetworkProcessConnectionParameters;
struct SharedPreferencesForWebProcess;
struct WebPushMessage;
struct WebsiteData;
}

namespace Messages {
namespace NetworkProcess {

static inline IPC::ReceiverName messageReceiverName()
{
    return IPC::ReceiverName::NetworkProcess;
}

class InitializeNetworkProcess {
public:
    using Arguments = std::tuple<WebKit::NetworkProcessCreationParameters>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_InitializeNetworkProcess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_InitializeNetworkProcessReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit InitializeNetworkProcess(WebKit::NetworkProcessCreationParameters&& processCreationParameters)
        : m_arguments(WTFMove(processCreationParameters))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::NetworkProcessCreationParameters&&> m_arguments;
};

class CreateNetworkConnectionToWebProcess {
public:
    using Arguments = std::tuple<WebCore::ProcessIdentifier, PAL::SessionID, WebKit::NetworkProcessConnectionParameters>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_CreateNetworkConnectionToWebProcess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_CreateNetworkConnectionToWebProcessReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<std::optional<IPC::ConnectionHandle>, WebCore::HTTPCookieAcceptPolicy>;
    using Reply = CompletionHandler<void(std::optional<IPC::ConnectionHandle>&&, WebCore::HTTPCookieAcceptPolicy)>;
    using Promise = WTF::NativePromise<std::tuple<std::optional<IPC::ConnectionHandle>, WebCore::HTTPCookieAcceptPolicy>, IPC::Error>;
    CreateNetworkConnectionToWebProcess(const WebCore::ProcessIdentifier& processIdentifier, const PAL::SessionID& sessionID, const WebKit::NetworkProcessConnectionParameters& parameters)
        : m_arguments(processIdentifier, sessionID, parameters)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ProcessIdentifier&, const PAL::SessionID&, const WebKit::NetworkProcessConnectionParameters&> m_arguments;
};

class SharedPreferencesForWebProcessDidChange {
public:
    using Arguments = std::tuple<WebCore::ProcessIdentifier, WebKit::SharedPreferencesForWebProcess>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SharedPreferencesForWebProcessDidChange; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SharedPreferencesForWebProcessDidChangeReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SharedPreferencesForWebProcessDidChange(const WebCore::ProcessIdentifier& processIdentifier, const WebKit::SharedPreferencesForWebProcess& sharedPreferencesForWebProcess)
        : m_arguments(processIdentifier, sharedPreferencesForWebProcess)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ProcessIdentifier&, const WebKit::SharedPreferencesForWebProcess&> m_arguments;
};

class AddAllowedFirstPartyForCookies {
public:
    using Arguments = std::tuple<WebCore::ProcessIdentifier, WebCore::RegistrableDomain, WebKit::LoadedWebArchive>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AddAllowedFirstPartyForCookies; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_AddAllowedFirstPartyForCookiesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    AddAllowedFirstPartyForCookies(const WebCore::ProcessIdentifier& processIdentifier, const WebCore::RegistrableDomain& firstPartyForCookies, WebKit::LoadedWebArchive loadedWebArchive)
        : m_arguments(processIdentifier, firstPartyForCookies, loadedWebArchive)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ProcessIdentifier&, const WebCore::RegistrableDomain&, WebKit::LoadedWebArchive> m_arguments;
};

#if USE(SOUP)
class SetIgnoreTLSErrors {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetIgnoreTLSErrors; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetIgnoreTLSErrors(const PAL::SessionID& sessionID, bool ignoreTLSErrors)
        : m_arguments(sessionID, ignoreTLSErrors)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};
#endif

#if USE(SOUP)
class UserPreferredLanguagesChanged {
public:
    using Arguments = std::tuple<Vector<String>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_UserPreferredLanguagesChanged; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit UserPreferredLanguagesChanged(const Vector<String>& languages)
        : m_arguments(languages)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Vector<String>&> m_arguments;
};
#endif

#if USE(SOUP)
class SetNetworkProxySettings {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::SoupNetworkProxySettings>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetNetworkProxySettings; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetNetworkProxySettings(const PAL::SessionID& sessionID, const WebCore::SoupNetworkProxySettings& settings)
        : m_arguments(sessionID, settings)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::SoupNetworkProxySettings&> m_arguments;
};
#endif

#if USE(SOUP)
class PrefetchDNS {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_PrefetchDNS; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit PrefetchDNS(const String& hostname)
        : m_arguments(hostname)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};
#endif

#if USE(SOUP)
class SetPersistentCredentialStorageEnabled {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPersistentCredentialStorageEnabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetPersistentCredentialStorageEnabled(const PAL::SessionID& sessionID, bool enabled)
        : m_arguments(sessionID, enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};
#endif

#if USE(CURL)
class SetNetworkProxySettings {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::CurlProxySettings>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetNetworkProxySettings; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetNetworkProxySettings(const PAL::SessionID& sessionID, const WebCore::CurlProxySettings& settings)
        : m_arguments(sessionID, settings)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::CurlProxySettings&> m_arguments;
};
#endif

class AddWebsiteDataStore {
public:
    using Arguments = std::tuple<WebKit::WebsiteDataStoreParameters>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AddWebsiteDataStore; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit AddWebsiteDataStore(WebKit::WebsiteDataStoreParameters&& websiteDataStoreParameters)
        : m_arguments(WTFMove(websiteDataStoreParameters))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::WebsiteDataStoreParameters&&> m_arguments;
};

class DestroySession {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DestroySession; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_DestroySessionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit DestroySession(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class FetchWebsiteData {
public:
    using Arguments = std::tuple<PAL::SessionID, OptionSet<WebKit::WebsiteDataType>, OptionSet<WebKit::WebsiteDataFetchOption>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_FetchWebsiteData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_FetchWebsiteDataReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<WebKit::WebsiteData>;
    using Reply = CompletionHandler<void(WebKit::WebsiteData&&)>;
    using Promise = WTF::NativePromise<WebKit::WebsiteData, IPC::Error>;
    FetchWebsiteData(const PAL::SessionID& sessionID, const OptionSet<WebKit::WebsiteDataType>& websiteDataTypes, const OptionSet<WebKit::WebsiteDataFetchOption>& fetchOptions)
        : m_arguments(sessionID, websiteDataTypes, fetchOptions)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const OptionSet<WebKit::WebsiteDataType>&, const OptionSet<WebKit::WebsiteDataFetchOption>&> m_arguments;
};

class DeleteWebsiteData {
public:
    using Arguments = std::tuple<PAL::SessionID, OptionSet<WebKit::WebsiteDataType>, WallTime>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DeleteWebsiteData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_DeleteWebsiteDataReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    DeleteWebsiteData(const PAL::SessionID& sessionID, const OptionSet<WebKit::WebsiteDataType>& websiteDataTypes, const WallTime& modifiedSince)
        : m_arguments(sessionID, websiteDataTypes, modifiedSince)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const OptionSet<WebKit::WebsiteDataType>&, const WallTime&> m_arguments;
};

class DeleteWebsiteDataForOrigins {
public:
    using Arguments = std::tuple<PAL::SessionID, OptionSet<WebKit::WebsiteDataType>, Vector<WebCore::SecurityOriginData>, Vector<String>, Vector<String>, Vector<WebCore::RegistrableDomain>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DeleteWebsiteDataForOrigins; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_DeleteWebsiteDataForOriginsReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    DeleteWebsiteDataForOrigins(const PAL::SessionID& sessionID, const OptionSet<WebKit::WebsiteDataType>& websiteDataTypes, const Vector<WebCore::SecurityOriginData>& origins, const Vector<String>& cookieHostNames, const Vector<String>& HSTSCacheHostNames, const Vector<WebCore::RegistrableDomain>& registrableDomains)
        : m_arguments(sessionID, websiteDataTypes, origins, cookieHostNames, HSTSCacheHostNames, registrableDomains)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const OptionSet<WebKit::WebsiteDataType>&, const Vector<WebCore::SecurityOriginData>&, const Vector<String>&, const Vector<String>&, const Vector<WebCore::RegistrableDomain>&> m_arguments;
};

class RenameOriginInWebsiteData {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::SecurityOriginData, WebCore::SecurityOriginData, OptionSet<WebKit::WebsiteDataType>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_RenameOriginInWebsiteData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_RenameOriginInWebsiteDataReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    RenameOriginInWebsiteData(const PAL::SessionID& sessionID, const WebCore::SecurityOriginData& oldOrigin, const WebCore::SecurityOriginData& newOrigin, const OptionSet<WebKit::WebsiteDataType>& websiteDataTypes)
        : m_arguments(sessionID, oldOrigin, newOrigin, websiteDataTypes)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::SecurityOriginData&, const WebCore::SecurityOriginData&, const OptionSet<WebKit::WebsiteDataType>&> m_arguments;
};

class WebsiteDataOriginDirectoryForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::ClientOrigin, OptionSet<WebKit::WebsiteDataType>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_WebsiteDataOriginDirectoryForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_WebsiteDataOriginDirectoryForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<String>;
    using Reply = CompletionHandler<void(String&&)>;
    using Promise = WTF::NativePromise<String, IPC::Error>;
    WebsiteDataOriginDirectoryForTesting(const PAL::SessionID& sessionID, const WebCore::ClientOrigin& origin, const OptionSet<WebKit::WebsiteDataType>& websiteDataType)
        : m_arguments(sessionID, origin, websiteDataType)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::ClientOrigin&, const OptionSet<WebKit::WebsiteDataType>&> m_arguments;
};

class DownloadRequest {
public:
    using Arguments = std::tuple<PAL::SessionID, WebKit::DownloadID, WebCore::ResourceRequest, std::optional<WebCore::SecurityOriginData>, std::optional<WebKit::NavigatingToAppBoundDomain>, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DownloadRequest; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    DownloadRequest(const PAL::SessionID& sessionID, const WebKit::DownloadID& downloadID, const WebCore::ResourceRequest& request, const std::optional<WebCore::SecurityOriginData>& topOrigin, const std::optional<WebKit::NavigatingToAppBoundDomain>& isNavigatingToAppBoundDomain, const String& suggestedFilename)
        : m_arguments(sessionID, downloadID, request, topOrigin, isNavigatingToAppBoundDomain, suggestedFilename)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebKit::DownloadID&, const WebCore::ResourceRequest&, const std::optional<WebCore::SecurityOriginData>&, const std::optional<WebKit::NavigatingToAppBoundDomain>&, const String&> m_arguments;
};

class ResumeDownload {
public:
    using Arguments = std::tuple<PAL::SessionID, WebKit::DownloadID, std::span<const uint8_t>, String, WebKit::SandboxExtensionHandle, WebKit::CallDownloadDidStart>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ResumeDownload; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    ResumeDownload(const PAL::SessionID& sessionID, const WebKit::DownloadID& downloadID, const std::span<const uint8_t>& resumeData, const String& path, WebKit::SandboxExtensionHandle&& sandboxExtensionHandle, WebKit::CallDownloadDidStart callDownloadDidStart)
        : m_arguments(sessionID, downloadID, resumeData, path, WTFMove(sandboxExtensionHandle), callDownloadDidStart)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebKit::DownloadID&, const std::span<const uint8_t>&, const String&, WebKit::SandboxExtensionHandle&&, WebKit::CallDownloadDidStart> m_arguments;
};

class CancelDownload {
public:
    using Arguments = std::tuple<WebKit::DownloadID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_CancelDownload; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_CancelDownloadReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<std::span<const uint8_t>>;
    using Reply = CompletionHandler<void(std::span<const uint8_t>&&)>;
    using Promise = WTF::NativePromise<std::span<const uint8_t>, IPC::Error>;
    explicit CancelDownload(const WebKit::DownloadID& downloadID)
        : m_arguments(downloadID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::DownloadID&> m_arguments;
};

#if HAVE(MODERN_DOWNLOADPROGRESS)
class PublishDownloadProgress {
public:
    using Arguments = std::tuple<WebKit::DownloadID, URL, std::span<const uint8_t>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_PublishDownloadProgress; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    PublishDownloadProgress(const WebKit::DownloadID& downloadID, const URL& url, const std::span<const uint8_t>& bookmarkData)
        : m_arguments(downloadID, url, bookmarkData)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::DownloadID&, const URL&, const std::span<const uint8_t>&> m_arguments;
};
#endif

#if PLATFORM(COCOA) && !HAVE(MODERN_DOWNLOADPROGRESS)
class PublishDownloadProgress {
public:
    using Arguments = std::tuple<WebKit::DownloadID, URL, WebKit::SandboxExtensionHandle>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_PublishDownloadProgress; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    PublishDownloadProgress(const WebKit::DownloadID& downloadID, const URL& url, WebKit::SandboxExtensionHandle&& sandboxExtensionHandle)
        : m_arguments(downloadID, url, WTFMove(sandboxExtensionHandle))
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::DownloadID&, const URL&, WebKit::SandboxExtensionHandle&&> m_arguments;
};
#endif

class DataTaskWithRequest {
public:
    using Arguments = std::tuple<WebKit::WebPageProxyIdentifier, PAL::SessionID, WebCore::ResourceRequest, std::optional<WebCore::SecurityOriginData>, IPC::FormDataReference>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DataTaskWithRequest; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_DataTaskWithRequestReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<WebKit::DataTaskIdentifier>;
    using Reply = CompletionHandler<void(WebKit::DataTaskIdentifier&&)>;
    using Promise = WTF::NativePromise<WebKit::DataTaskIdentifier, IPC::Error>;
    DataTaskWithRequest(const WebKit::WebPageProxyIdentifier& pageID, const PAL::SessionID& sessionID, const WebCore::ResourceRequest& request, const std::optional<WebCore::SecurityOriginData>& topOrigin, const IPC::FormDataReference& requestBody)
        : m_arguments(pageID, sessionID, request, topOrigin, requestBody)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::WebPageProxyIdentifier&, const PAL::SessionID&, const WebCore::ResourceRequest&, const std::optional<WebCore::SecurityOriginData>&, const IPC::FormDataReference&> m_arguments;
};

class CancelDataTask {
public:
    using Arguments = std::tuple<WebKit::DataTaskIdentifier, PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_CancelDataTask; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_CancelDataTaskReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    CancelDataTask(const WebKit::DataTaskIdentifier& taskIdentifier, const PAL::SessionID& sessionID)
        : m_arguments(taskIdentifier, sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::DataTaskIdentifier&, const PAL::SessionID&> m_arguments;
};

class ApplicationDidEnterBackground {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ApplicationDidEnterBackground; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class ApplicationWillEnterForeground {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ApplicationWillEnterForeground; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class FlushCookies {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_FlushCookies; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_FlushCookiesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit FlushCookies(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

#if !PLATFORM(COCOA)
class AllowSpecificHTTPSCertificateForHost {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::CertificateInfo, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AllowSpecificHTTPSCertificateForHost; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    AllowSpecificHTTPSCertificateForHost(const PAL::SessionID& sessionID, const WebCore::CertificateInfo& certificate, const String& host)
        : m_arguments(sessionID, certificate, host)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::CertificateInfo&, const String&> m_arguments;
};
#endif

class AllowTLSCertificateChainForLocalPCMTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::CertificateInfo>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AllowTLSCertificateChainForLocalPCMTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    AllowTLSCertificateChainForLocalPCMTesting(const PAL::SessionID& sessionID, const WebCore::CertificateInfo& certificate)
        : m_arguments(sessionID, certificate)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::CertificateInfo&> m_arguments;
};

class SetCacheModel {
public:
    using Arguments = std::tuple<WebKit::CacheModel>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetCacheModel; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetCacheModel(WebKit::CacheModel cacheModel)
        : m_arguments(cacheModel)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::CacheModel> m_arguments;
};

class SetCacheModelSynchronouslyForTesting {
public:
    using Arguments = std::tuple<WebKit::CacheModel>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetCacheModelSynchronouslyForTesting; }
    static constexpr bool isSync = true;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    explicit SetCacheModelSynchronouslyForTesting(WebKit::CacheModel cacheModel)
        : m_arguments(cacheModel)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::CacheModel> m_arguments;
};

class ProcessWillSuspendImminentlyForTestingSync {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ProcessWillSuspendImminentlyForTestingSync; }
    static constexpr bool isSync = true;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class PrepareToSuspend {
public:
    using Arguments = std::tuple<bool, MonotonicTime>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_PrepareToSuspend; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_PrepareToSuspendReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    PrepareToSuspend(bool isSuspensionImminent, const MonotonicTime& estimatedSuspendTime)
        : m_arguments(isSuspensionImminent, estimatedSuspendTime)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool, const MonotonicTime&> m_arguments;
};

class ProcessDidResume {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ProcessDidResume; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit ProcessDidResume(bool forForegroundActivity)
        : m_arguments(forForegroundActivity)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class NotifyMediaStreamingActivity {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_NotifyMediaStreamingActivity; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit NotifyMediaStreamingActivity(bool activity)
        : m_arguments(activity)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class PreconnectTo {
public:
    using Arguments = std::tuple<PAL::SessionID, WebKit::WebPageProxyIdentifier, WebCore::PageIdentifier, WebCore::ResourceRequest, WebCore::StoredCredentialsPolicy, std::optional<WebKit::NavigatingToAppBoundDomain>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_PreconnectTo; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    PreconnectTo(const PAL::SessionID& sessionID, const WebKit::WebPageProxyIdentifier& webPageProxyID, const WebCore::PageIdentifier& webPageID, const WebCore::ResourceRequest& request, WebCore::StoredCredentialsPolicy storedCredentialsPolicy, const std::optional<WebKit::NavigatingToAppBoundDomain>& isNavigatingToAppBoundDomain)
        : m_arguments(sessionID, webPageProxyID, webPageID, request, storedCredentialsPolicy, isNavigatingToAppBoundDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebKit::WebPageProxyIdentifier&, const WebCore::PageIdentifier&, const WebCore::ResourceRequest&, WebCore::StoredCredentialsPolicy, const std::optional<WebKit::NavigatingToAppBoundDomain>&> m_arguments;
};

class SetInspectionForServiceWorkersAllowed {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetInspectionForServiceWorkersAllowed; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetInspectionForServiceWorkersAllowed(const PAL::SessionID& sessionID, bool inspectable)
        : m_arguments(sessionID, inspectable)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class ClearPrevalentResource {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClearPrevalentResource; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClearPrevalentResourceReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    ClearPrevalentResource(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain)
        : m_arguments(sessionID, resourceDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class ClearUserInteraction {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClearUserInteraction; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClearUserInteractionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    ClearUserInteraction(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain)
        : m_arguments(sessionID, resourceDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class DumpResourceLoadStatistics {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DumpResourceLoadStatistics; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_DumpResourceLoadStatisticsReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<String>;
    using Reply = CompletionHandler<void(String&&)>;
    using Promise = WTF::NativePromise<String, IPC::Error>;
    explicit DumpResourceLoadStatistics(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetTrackingPreventionEnabled {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetTrackingPreventionEnabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetTrackingPreventionEnabled(const PAL::SessionID& sessionID, bool enabled)
        : m_arguments(sessionID, enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class UpdateStorageAccessPromptQuirks {
public:
    using Arguments = std::tuple<Vector<WebCore::OrganizationStorageAccessPromptQuirk>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_UpdateStorageAccessPromptQuirks; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit UpdateStorageAccessPromptQuirks(const Vector<WebCore::OrganizationStorageAccessPromptQuirk>& organizationStorageAccessQuirks)
        : m_arguments(organizationStorageAccessQuirks)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Vector<WebCore::OrganizationStorageAccessPromptQuirk>&> m_arguments;
};

class SetResourceLoadStatisticsLogTestingEvent {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetResourceLoadStatisticsLogTestingEvent; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit SetResourceLoadStatisticsLogTestingEvent(bool enabled)
        : m_arguments(enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class UpdatePrevalentDomainsToBlockCookiesFor {
public:
    using Arguments = std::tuple<PAL::SessionID, Vector<WebCore::RegistrableDomain>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_UpdatePrevalentDomainsToBlockCookiesFor; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_UpdatePrevalentDomainsToBlockCookiesForReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    UpdatePrevalentDomainsToBlockCookiesFor(const PAL::SessionID& sessionID, const Vector<WebCore::RegistrableDomain>& domainsToBlock)
        : m_arguments(sessionID, domainsToBlock)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Vector<WebCore::RegistrableDomain>&> m_arguments;
};

class IsGrandfathered {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_IsGrandfathered; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_IsGrandfatheredReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    IsGrandfathered(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& targetDomain)
        : m_arguments(sessionID, targetDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class IsPrevalentResource {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_IsPrevalentResource; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_IsPrevalentResourceReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    IsPrevalentResource(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& targetDomain)
        : m_arguments(sessionID, targetDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class IsVeryPrevalentResource {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_IsVeryPrevalentResource; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_IsVeryPrevalentResourceReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    IsVeryPrevalentResource(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& targetDomain)
        : m_arguments(sessionID, targetDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetLastSeen {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetLastSeen; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetLastSeenReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetLastSeen(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain, const Seconds& seconds)
        : m_arguments(sessionID, resourceDomain, seconds)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const Seconds&> m_arguments;
};

class MergeStatisticForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain, WebCore::RegistrableDomain, Seconds, bool, Seconds, bool, bool, bool, uint64_t>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_MergeStatisticForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_MergeStatisticForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    MergeStatisticForTesting(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain, const WebCore::RegistrableDomain& topFrameDomain1, const WebCore::RegistrableDomain& topFrameDomain2, const Seconds& lastSeen, bool hadUserInteraction, const Seconds& mostRecentUserInteraction, bool isGrandfathered, bool isPrevalent, bool isVeryPrevalent, uint64_t dataRecordsRemoved)
        : m_arguments(sessionID, resourceDomain, topFrameDomain1, topFrameDomain2, lastSeen, hadUserInteraction, mostRecentUserInteraction, isGrandfathered, isPrevalent, isVeryPrevalent, dataRecordsRemoved)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&, const Seconds&, bool, const Seconds&, bool, bool, bool, uint64_t> m_arguments;
};

class InsertExpiredStatisticForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, uint64_t, bool, bool, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_InsertExpiredStatisticForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_InsertExpiredStatisticForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    InsertExpiredStatisticForTesting(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain, uint64_t numberOfOperatingDaysPassed, bool hadUserInteraction, bool isScheduledForAllButCookieDataRemoval, bool isPrevalent)
        : m_arguments(sessionID, resourceDomain, numberOfOperatingDaysPassed, hadUserInteraction, isScheduledForAllButCookieDataRemoval, isPrevalent)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, uint64_t, bool, bool, bool> m_arguments;
};

class SetPrevalentResource {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrevalentResource; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPrevalentResourceReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPrevalentResource(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain)
        : m_arguments(sessionID, resourceDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetPrevalentResourceForDebugMode {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrevalentResourceForDebugMode; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPrevalentResourceForDebugModeReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPrevalentResourceForDebugMode(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain)
        : m_arguments(sessionID, resourceDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class IsResourceLoadStatisticsEphemeral {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_IsResourceLoadStatisticsEphemeral; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_IsResourceLoadStatisticsEphemeralReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    explicit IsResourceLoadStatisticsEphemeral(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class HadUserInteraction {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_HadUserInteraction; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_HadUserInteractionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    HadUserInteraction(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain)
        : m_arguments(sessionID, resourceDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class IsRelationshipOnlyInDatabaseOnce {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_IsRelationshipOnlyInDatabaseOnce; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_IsRelationshipOnlyInDatabaseOnceReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    IsRelationshipOnlyInDatabaseOnce(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& subDomain, const WebCore::RegistrableDomain& topDomain)
        : m_arguments(sessionID, subDomain, topDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class HasLocalStorage {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_HasLocalStorage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_HasLocalStorageReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    HasLocalStorage(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain)
        : m_arguments(sessionID, resourceDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class GetAllStorageAccessEntries {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_GetAllStorageAccessEntries; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_GetAllStorageAccessEntriesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<Vector<String>>;
    using Reply = CompletionHandler<void(Vector<String>&&)>;
    using Promise = WTF::NativePromise<Vector<String>, IPC::Error>;
    explicit GetAllStorageAccessEntries(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class IsRegisteredAsRedirectingTo {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_IsRegisteredAsRedirectingTo; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_IsRegisteredAsRedirectingToReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    IsRegisteredAsRedirectingTo(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& redirectedFromDomain, const WebCore::RegistrableDomain& redirectedToDomain)
        : m_arguments(sessionID, redirectedFromDomain, redirectedToDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class IsRegisteredAsSubFrameUnder {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_IsRegisteredAsSubFrameUnder; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_IsRegisteredAsSubFrameUnderReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    IsRegisteredAsSubFrameUnder(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& subFrameDomain, const WebCore::RegistrableDomain& topFrameDomain)
        : m_arguments(sessionID, subFrameDomain, topFrameDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class IsRegisteredAsSubresourceUnder {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_IsRegisteredAsSubresourceUnder; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_IsRegisteredAsSubresourceUnderReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    IsRegisteredAsSubresourceUnder(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& subresourceDomain, const WebCore::RegistrableDomain& topFrameDomain)
        : m_arguments(sessionID, subresourceDomain, topFrameDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class DomainIDExistsInDatabase {
public:
    using Arguments = std::tuple<PAL::SessionID, int>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DomainIDExistsInDatabase; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_DomainIDExistsInDatabaseReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    DomainIDExistsInDatabase(const PAL::SessionID& sessionID, const int& domainID)
        : m_arguments(sessionID, domainID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const int&> m_arguments;
};

class LogFrameNavigation {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain, WebCore::RegistrableDomain, bool, bool, Seconds, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_LogFrameNavigation; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    LogFrameNavigation(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& targetDomain, const WebCore::RegistrableDomain& topFrameDomain, const WebCore::RegistrableDomain& sourceDomain, bool isRedirect, bool isMainFrame, const Seconds& delayAfterMainFrameDocumentLoad, bool wasPotentiallyInitiatedByUser)
        : m_arguments(sessionID, targetDomain, topFrameDomain, sourceDomain, isRedirect, isMainFrame, delayAfterMainFrameDocumentLoad, wasPotentiallyInitiatedByUser)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&, bool, bool, const Seconds&, bool> m_arguments;
};

class LogUserInteraction {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_LogUserInteraction; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_LogUserInteractionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    LogUserInteraction(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& topFrameDomain)
        : m_arguments(sessionID, topFrameDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class ResetParametersToDefaultValues {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ResetParametersToDefaultValues; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ResetParametersToDefaultValuesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ResetParametersToDefaultValues(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class ScheduleClearInMemoryAndPersistent {
public:
    using Arguments = std::tuple<PAL::SessionID, std::optional<WallTime>, WebKit::ShouldGrandfatherStatistics>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ScheduleClearInMemoryAndPersistent; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ScheduleClearInMemoryAndPersistentReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    ScheduleClearInMemoryAndPersistent(const PAL::SessionID& sessionID, const std::optional<WallTime>& modifiedSince, WebKit::ShouldGrandfatherStatistics shouldGrandfather)
        : m_arguments(sessionID, modifiedSince, shouldGrandfather)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const std::optional<WallTime>&, WebKit::ShouldGrandfatherStatistics> m_arguments;
};

class ScheduleCookieBlockingUpdate {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ScheduleCookieBlockingUpdate; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ScheduleCookieBlockingUpdateReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ScheduleCookieBlockingUpdate(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class ScheduleStatisticsAndDataRecordsProcessing {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ScheduleStatisticsAndDataRecordsProcessing; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ScheduleStatisticsAndDataRecordsProcessingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ScheduleStatisticsAndDataRecordsProcessing(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class StatisticsDatabaseHasAllTables {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_StatisticsDatabaseHasAllTables; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_StatisticsDatabaseHasAllTablesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    explicit StatisticsDatabaseHasAllTables(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetCacheMaxAgeCapForPrevalentResources {
public:
    using Arguments = std::tuple<PAL::SessionID, Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetCacheMaxAgeCapForPrevalentResources; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetCacheMaxAgeCapForPrevalentResourcesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetCacheMaxAgeCapForPrevalentResources(const PAL::SessionID& sessionID, const Seconds& seconds)
        : m_arguments(sessionID, seconds)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Seconds&> m_arguments;
};

class SetGrandfathered {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetGrandfathered; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetGrandfatheredReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetGrandfathered(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain, bool isGrandfathered)
        : m_arguments(sessionID, resourceDomain, isGrandfathered)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, bool> m_arguments;
};

class GetResourceLoadStatisticsDataSummary {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_GetResourceLoadStatisticsDataSummary; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_GetResourceLoadStatisticsDataSummaryReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<Vector<WebKit::ITPThirdPartyData>>;
    using Reply = CompletionHandler<void(Vector<WebKit::ITPThirdPartyData>&&)>;
    using Promise = WTF::NativePromise<Vector<WebKit::ITPThirdPartyData>, IPC::Error>;
    explicit GetResourceLoadStatisticsDataSummary(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetGrandfatheringTime {
public:
    using Arguments = std::tuple<PAL::SessionID, Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetGrandfatheringTime; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetGrandfatheringTimeReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetGrandfatheringTime(const PAL::SessionID& sessionID, const Seconds& seconds)
        : m_arguments(sessionID, seconds)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Seconds&> m_arguments;
};

class SetMaxStatisticsEntries {
public:
    using Arguments = std::tuple<PAL::SessionID, uint64_t>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetMaxStatisticsEntries; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetMaxStatisticsEntriesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetMaxStatisticsEntries(const PAL::SessionID& sessionID, uint64_t maximumEntryCount)
        : m_arguments(sessionID, maximumEntryCount)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, uint64_t> m_arguments;
};

class SetMinimumTimeBetweenDataRecordsRemoval {
public:
    using Arguments = std::tuple<PAL::SessionID, Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetMinimumTimeBetweenDataRecordsRemoval; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetMinimumTimeBetweenDataRecordsRemovalReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetMinimumTimeBetweenDataRecordsRemoval(const PAL::SessionID& sessionID, const Seconds& seconds)
        : m_arguments(sessionID, seconds)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Seconds&> m_arguments;
};

class SetPruneEntriesDownTo {
public:
    using Arguments = std::tuple<PAL::SessionID, uint64_t>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPruneEntriesDownTo; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPruneEntriesDownToReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPruneEntriesDownTo(const PAL::SessionID& sessionID, uint64_t maximumEntryCount)
        : m_arguments(sessionID, maximumEntryCount)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, uint64_t> m_arguments;
};

class SetShouldClassifyResourcesBeforeDataRecordsRemoval {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetShouldClassifyResourcesBeforeDataRecordsRemoval; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetShouldClassifyResourcesBeforeDataRecordsRemovalReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetShouldClassifyResourcesBeforeDataRecordsRemoval(const PAL::SessionID& sessionID, bool value)
        : m_arguments(sessionID, value)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SetNotifyPagesWhenDataRecordsWereScanned {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetNotifyPagesWhenDataRecordsWereScanned; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetNotifyPagesWhenDataRecordsWereScannedReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetNotifyPagesWhenDataRecordsWereScanned(const PAL::SessionID& sessionID, bool value)
        : m_arguments(sessionID, value)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SetResourceLoadStatisticsTimeAdvanceForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetResourceLoadStatisticsTimeAdvanceForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetResourceLoadStatisticsTimeAdvanceForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetResourceLoadStatisticsTimeAdvanceForTesting(const PAL::SessionID& sessionID, const Seconds& time)
        : m_arguments(sessionID, time)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Seconds&> m_arguments;
};

class SetIsRunningResourceLoadStatisticsTest {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetIsRunningResourceLoadStatisticsTest; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetIsRunningResourceLoadStatisticsTestReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetIsRunningResourceLoadStatisticsTest(const PAL::SessionID& sessionID, bool value)
        : m_arguments(sessionID, value)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SetResourceLoadStatisticsDebugMode {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetResourceLoadStatisticsDebugMode; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetResourceLoadStatisticsDebugModeReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetResourceLoadStatisticsDebugMode(const PAL::SessionID& sessionID, bool debugMode)
        : m_arguments(sessionID, debugMode)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SetVeryPrevalentResource {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetVeryPrevalentResource; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetVeryPrevalentResourceReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetVeryPrevalentResource(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& resourceDomain)
        : m_arguments(sessionID, resourceDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetSubframeUnderTopFrameDomain {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetSubframeUnderTopFrameDomain; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetSubframeUnderTopFrameDomainReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetSubframeUnderTopFrameDomain(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& subFrameDomain, const WebCore::RegistrableDomain& topFrameDomain)
        : m_arguments(sessionID, subFrameDomain, topFrameDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetSubresourceUnderTopFrameDomain {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetSubresourceUnderTopFrameDomain; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetSubresourceUnderTopFrameDomainReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetSubresourceUnderTopFrameDomain(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& subresourceDomain, const WebCore::RegistrableDomain& topFrameDomain)
        : m_arguments(sessionID, subresourceDomain, topFrameDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetSubresourceUniqueRedirectTo {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetSubresourceUniqueRedirectTo; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetSubresourceUniqueRedirectToReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetSubresourceUniqueRedirectTo(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& subresourceDomain, const WebCore::RegistrableDomain& redirectedToDomain)
        : m_arguments(sessionID, subresourceDomain, redirectedToDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetSubresourceUniqueRedirectFrom {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetSubresourceUniqueRedirectFrom; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetSubresourceUniqueRedirectFromReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetSubresourceUniqueRedirectFrom(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& subresourceDomain, const WebCore::RegistrableDomain& redirectedFromDomain)
        : m_arguments(sessionID, subresourceDomain, redirectedFromDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetTimeToLiveUserInteraction {
public:
    using Arguments = std::tuple<PAL::SessionID, Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetTimeToLiveUserInteraction; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetTimeToLiveUserInteractionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetTimeToLiveUserInteraction(const PAL::SessionID& sessionID, const Seconds& seconds)
        : m_arguments(sessionID, seconds)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Seconds&> m_arguments;
};

class SetTopFrameUniqueRedirectTo {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetTopFrameUniqueRedirectTo; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetTopFrameUniqueRedirectToReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetTopFrameUniqueRedirectTo(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& topFrameDomain, const WebCore::RegistrableDomain& redirectedToDomain)
        : m_arguments(sessionID, topFrameDomain, redirectedToDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetTopFrameUniqueRedirectFrom {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetTopFrameUniqueRedirectFrom; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetTopFrameUniqueRedirectFromReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetTopFrameUniqueRedirectFrom(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& topFrameDomain, const WebCore::RegistrableDomain& redirectedFromDomain)
        : m_arguments(sessionID, topFrameDomain, redirectedFromDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&> m_arguments;
};

class ResetCacheMaxAgeCapForPrevalentResources {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ResetCacheMaxAgeCapForPrevalentResources; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ResetCacheMaxAgeCapForPrevalentResourcesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ResetCacheMaxAgeCapForPrevalentResources(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class DidCommitCrossSiteLoadWithDataTransfer {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain, OptionSet<WebCore::CrossSiteNavigationDataTransferFlag>, WebKit::WebPageProxyIdentifier, WebCore::PageIdentifier, WebKit::DidFilterKnownLinkDecoration>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DidCommitCrossSiteLoadWithDataTransfer; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    DidCommitCrossSiteLoadWithDataTransfer(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& fromDomain, const WebCore::RegistrableDomain& toDomain, const OptionSet<WebCore::CrossSiteNavigationDataTransferFlag>& navigationDataTransfer, const WebKit::WebPageProxyIdentifier& webPageProxyID, const WebCore::PageIdentifier& webPageID, WebKit::DidFilterKnownLinkDecoration didFilterKnownLinkDecoration)
        : m_arguments(sessionID, fromDomain, toDomain, navigationDataTransfer, webPageProxyID, webPageID, didFilterKnownLinkDecoration)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&, const OptionSet<WebCore::CrossSiteNavigationDataTransferFlag>&, const WebKit::WebPageProxyIdentifier&, const WebCore::PageIdentifier&, WebKit::DidFilterKnownLinkDecoration> m_arguments;
};

class SetCrossSiteLoadWithLinkDecorationForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, WebCore::RegistrableDomain, WebKit::DidFilterKnownLinkDecoration>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetCrossSiteLoadWithLinkDecorationForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetCrossSiteLoadWithLinkDecorationForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetCrossSiteLoadWithLinkDecorationForTesting(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& fromDomain, const WebCore::RegistrableDomain& toDomain, WebKit::DidFilterKnownLinkDecoration didFilterKnownLinkDecoration)
        : m_arguments(sessionID, fromDomain, toDomain, didFilterKnownLinkDecoration)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::RegistrableDomain&, WebKit::DidFilterKnownLinkDecoration> m_arguments;
};

class ResetCrossSiteLoadsWithLinkDecorationForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ResetCrossSiteLoadsWithLinkDecorationForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ResetCrossSiteLoadsWithLinkDecorationForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ResetCrossSiteLoadsWithLinkDecorationForTesting(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class GrantStorageAccessForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, Vector<WebCore::RegistrableDomain>, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_GrantStorageAccessForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_GrantStorageAccessForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    GrantStorageAccessForTesting(const PAL::SessionID& sessionID, const Vector<WebCore::RegistrableDomain>& subFrameDomain, const WebCore::RegistrableDomain& topFrameDomain)
        : m_arguments(sessionID, subFrameDomain, topFrameDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Vector<WebCore::RegistrableDomain>&, const WebCore::RegistrableDomain&> m_arguments;
};

class DeleteCookiesForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DeleteCookiesForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_DeleteCookiesForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    DeleteCookiesForTesting(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& domain, bool includeHttpOnlyCookies)
        : m_arguments(sessionID, domain, includeHttpOnlyCookies)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&, bool> m_arguments;
};

class HasIsolatedSession {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_HasIsolatedSession; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_HasIsolatedSessionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    HasIsolatedSession(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& domain)
        : m_arguments(sessionID, domain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class CloseITPDatabase {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_CloseITPDatabase; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_CloseITPDatabaseReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit CloseITPDatabase(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

#if ENABLE(APP_BOUND_DOMAINS)
class SetAppBoundDomainsForResourceLoadStatistics {
public:
    using Arguments = std::tuple<PAL::SessionID, HashSet<WebCore::RegistrableDomain>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetAppBoundDomainsForResourceLoadStatistics; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetAppBoundDomainsForResourceLoadStatisticsReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetAppBoundDomainsForResourceLoadStatistics(const PAL::SessionID& sessionID, const HashSet<WebCore::RegistrableDomain>& appBoundDomains)
        : m_arguments(sessionID, appBoundDomains)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const HashSet<WebCore::RegistrableDomain>&> m_arguments;
};
#endif

#if ENABLE(MANAGED_DOMAINS)
class SetManagedDomainsForResourceLoadStatistics {
public:
    using Arguments = std::tuple<PAL::SessionID, HashSet<WebCore::RegistrableDomain>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetManagedDomainsForResourceLoadStatistics; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetManagedDomainsForResourceLoadStatisticsReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetManagedDomainsForResourceLoadStatistics(const PAL::SessionID& sessionID, const HashSet<WebCore::RegistrableDomain>& appBoundDomains)
        : m_arguments(sessionID, appBoundDomains)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const HashSet<WebCore::RegistrableDomain>&> m_arguments;
};
#endif

class SetShouldDowngradeReferrerForTesting {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetShouldDowngradeReferrerForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetShouldDowngradeReferrerForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit SetShouldDowngradeReferrerForTesting(bool enabled)
        : m_arguments(enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<bool> m_arguments;
};

class SetThirdPartyCookieBlockingMode {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::ThirdPartyCookieBlockingMode>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetThirdPartyCookieBlockingMode; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetThirdPartyCookieBlockingModeReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetThirdPartyCookieBlockingMode(const PAL::SessionID& sessionID, WebCore::ThirdPartyCookieBlockingMode blockingMode)
        : m_arguments(sessionID, blockingMode)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, WebCore::ThirdPartyCookieBlockingMode> m_arguments;
};

class SetShouldEnbleSameSiteStrictEnforcementForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::SameSiteStrictEnforcementEnabled>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetShouldEnbleSameSiteStrictEnforcementForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetShouldEnbleSameSiteStrictEnforcementForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetShouldEnbleSameSiteStrictEnforcementForTesting(const PAL::SessionID& sessionID, WebCore::SameSiteStrictEnforcementEnabled enabled)
        : m_arguments(sessionID, enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, WebCore::SameSiteStrictEnforcementEnabled> m_arguments;
};

class SetFirstPartyWebsiteDataRemovalModeForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::FirstPartyWebsiteDataRemovalMode>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetFirstPartyWebsiteDataRemovalModeForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetFirstPartyWebsiteDataRemovalModeForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetFirstPartyWebsiteDataRemovalModeForTesting(const PAL::SessionID& sessionID, WebCore::FirstPartyWebsiteDataRemovalMode mode)
        : m_arguments(sessionID, mode)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, WebCore::FirstPartyWebsiteDataRemovalMode> m_arguments;
};

class SetToSameSiteStrictCookiesForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetToSameSiteStrictCookiesForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetToSameSiteStrictCookiesForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetToSameSiteStrictCookiesForTesting(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& domain)
        : m_arguments(sessionID, domain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetFirstPartyHostCNAMEDomainForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, String, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetFirstPartyHostCNAMEDomainForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetFirstPartyHostCNAMEDomainForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetFirstPartyHostCNAMEDomainForTesting(const PAL::SessionID& sessionID, const String& firstPartyHost, const WebCore::RegistrableDomain& cnameDomain)
        : m_arguments(sessionID, firstPartyHost, cnameDomain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const String&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetThirdPartyCNAMEDomainForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::RegistrableDomain>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetThirdPartyCNAMEDomainForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetThirdPartyCNAMEDomainForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetThirdPartyCNAMEDomainForTesting(const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& domain)
        : m_arguments(sessionID, domain)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::RegistrableDomain&> m_arguments;
};

class SetPrivateClickMeasurementDebugMode {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementDebugMode; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetPrivateClickMeasurementDebugMode(const PAL::SessionID& sessionID, bool debugMode)
        : m_arguments(sessionID, debugMode)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SetBlobRegistryTopOriginPartitioningEnabled {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetBlobRegistryTopOriginPartitioningEnabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetBlobRegistryTopOriginPartitioningEnabled(const PAL::SessionID& sessionID, bool enabled)
        : m_arguments(sessionID, enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SetShouldSendPrivateTokenIPCForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetShouldSendPrivateTokenIPCForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetShouldSendPrivateTokenIPCForTesting(const PAL::SessionID& sessionID, bool enabled)
        : m_arguments(sessionID, enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SetSessionIsControlledByAutomation {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetSessionIsControlledByAutomation; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetSessionIsControlledByAutomation(const PAL::SessionID& sessionID, bool controlled)
        : m_arguments(sessionID, controlled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class RegisterURLSchemeAsSecure {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_RegisterURLSchemeAsSecure; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsSecure(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsBypassingContentSecurityPolicy {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_RegisterURLSchemeAsBypassingContentSecurityPolicy; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsBypassingContentSecurityPolicy(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsLocal {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_RegisterURLSchemeAsLocal; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsLocal(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class RegisterURLSchemeAsNoAccess {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_RegisterURLSchemeAsNoAccess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit RegisterURLSchemeAsNoAccess(const String& scheme)
        : m_arguments(scheme)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class SyncLocalStorage {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SyncLocalStorage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SyncLocalStorageReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class StoreServiceWorkerRegistrations {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_StoreServiceWorkerRegistrations; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_StoreServiceWorkerRegistrationsReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit StoreServiceWorkerRegistrations(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class StorePrivateClickMeasurement {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::PrivateClickMeasurement>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_StorePrivateClickMeasurement; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    StorePrivateClickMeasurement(const PAL::SessionID& sessionID, const WebCore::PrivateClickMeasurement& privateClickMeasurement)
        : m_arguments(sessionID, privateClickMeasurement)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::PrivateClickMeasurement&> m_arguments;
};

class DumpPrivateClickMeasurement {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DumpPrivateClickMeasurement; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_DumpPrivateClickMeasurementReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<String>;
    using Reply = CompletionHandler<void(String&&)>;
    using Promise = WTF::NativePromise<String, IPC::Error>;
    explicit DumpPrivateClickMeasurement(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class ClearPrivateClickMeasurement {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClearPrivateClickMeasurement; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClearPrivateClickMeasurementReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ClearPrivateClickMeasurement(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetPrivateClickMeasurementOverrideTimerForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementOverrideTimerForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementOverrideTimerForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPrivateClickMeasurementOverrideTimerForTesting(const PAL::SessionID& sessionID, bool value)
        : m_arguments(sessionID, value)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class MarkAttributedPrivateClickMeasurementsAsExpiredForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_MarkAttributedPrivateClickMeasurementsAsExpiredForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_MarkAttributedPrivateClickMeasurementsAsExpiredForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit MarkAttributedPrivateClickMeasurementsAsExpiredForTesting(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetPrivateClickMeasurementEphemeralMeasurementForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementEphemeralMeasurementForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementEphemeralMeasurementForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPrivateClickMeasurementEphemeralMeasurementForTesting(const PAL::SessionID& sessionID, bool value)
        : m_arguments(sessionID, value)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SimulatePrivateClickMeasurementSessionRestart {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SimulatePrivateClickMeasurementSessionRestart; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SimulatePrivateClickMeasurementSessionRestartReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit SimulatePrivateClickMeasurementSessionRestart(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetPrivateClickMeasurementTokenPublicKeyURLForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, URL>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementTokenPublicKeyURLForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementTokenPublicKeyURLForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPrivateClickMeasurementTokenPublicKeyURLForTesting(const PAL::SessionID& sessionID, const URL& url)
        : m_arguments(sessionID, url)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const URL&> m_arguments;
};

class SetPrivateClickMeasurementTokenSignatureURLForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, URL>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementTokenSignatureURLForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementTokenSignatureURLForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPrivateClickMeasurementTokenSignatureURLForTesting(const PAL::SessionID& sessionID, const URL& url)
        : m_arguments(sessionID, url)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const URL&> m_arguments;
};

class SetPrivateClickMeasurementAttributionReportURLsForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, URL, URL>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementAttributionReportURLsForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementAttributionReportURLsForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPrivateClickMeasurementAttributionReportURLsForTesting(const PAL::SessionID& sessionID, const URL& sourceURL, const URL& destinationURL)
        : m_arguments(sessionID, sourceURL, destinationURL)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const URL&, const URL&> m_arguments;
};

class MarkPrivateClickMeasurementsAsExpiredForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_MarkPrivateClickMeasurementsAsExpiredForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_MarkPrivateClickMeasurementsAsExpiredForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit MarkPrivateClickMeasurementsAsExpiredForTesting(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetPCMFraudPreventionValuesForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, String, String, String, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPCMFraudPreventionValuesForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPCMFraudPreventionValuesForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPCMFraudPreventionValuesForTesting(const PAL::SessionID& sessionID, const String& secretToken, const String& unlinkableToken, const String& signature, const String& keyID)
        : m_arguments(sessionID, secretToken, unlinkableToken, signature, keyID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const String&, const String&, const String&, const String&> m_arguments;
};

class SetPrivateClickMeasurementAppBundleIDForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementAppBundleIDForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPrivateClickMeasurementAppBundleIDForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPrivateClickMeasurementAppBundleIDForTesting(const PAL::SessionID& sessionID, const String& appBundleIDForTesting)
        : m_arguments(sessionID, appBundleIDForTesting)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const String&> m_arguments;
};

class ClosePCMDatabase {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClosePCMDatabase; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClosePCMDatabaseReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ClosePCMDatabase(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetServiceWorkerFetchTimeoutForTesting {
public:
    using Arguments = std::tuple<Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetServiceWorkerFetchTimeoutForTesting; }
    static constexpr bool isSync = true;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    explicit SetServiceWorkerFetchTimeoutForTesting(const Seconds& seconds)
        : m_arguments(seconds)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Seconds&> m_arguments;
};

class ResetServiceWorkerFetchTimeoutForTesting {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ResetServiceWorkerFetchTimeoutForTesting; }
    static constexpr bool isSync = true;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class ResetQuota {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ResetQuota; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ResetQuotaReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ResetQuota(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class SetOriginQuotaRatioEnabledForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetOriginQuotaRatioEnabledForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetOriginQuotaRatioEnabledForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetOriginQuotaRatioEnabledForTesting(const PAL::SessionID& sessionID, bool enabled)
        : m_arguments(sessionID, enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

#if PLATFORM(IOS_FAMILY)
class SetBackupExclusionPeriodForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, Seconds>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetBackupExclusionPeriodForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetBackupExclusionPeriodForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetBackupExclusionPeriodForTesting(const PAL::SessionID& sessionID, const Seconds& period)
        : m_arguments(sessionID, period)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Seconds&> m_arguments;
};
#endif

class ResetStoragePersistedState {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ResetStoragePersistedState; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ResetStoragePersistedStateReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ResetStoragePersistedState(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class CloneSessionStorageForWebPage {
public:
    using Arguments = std::tuple<PAL::SessionID, WebKit::WebPageProxyIdentifier, WebKit::WebPageProxyIdentifier>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_CloneSessionStorageForWebPage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    CloneSessionStorageForWebPage(const PAL::SessionID& sessionID, const WebKit::WebPageProxyIdentifier& sourcePage, const WebKit::WebPageProxyIdentifier& destinationPage)
        : m_arguments(sessionID, sourcePage, destinationPage)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebKit::WebPageProxyIdentifier&, const WebKit::WebPageProxyIdentifier&> m_arguments;
};

class DidIncreaseQuota {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::ClientOrigin, WebKit::QuotaIncreaseRequestIdentifier, std::optional<uint64_t>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DidIncreaseQuota; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    DidIncreaseQuota(const PAL::SessionID& sessionID, const WebCore::ClientOrigin& origin, const WebKit::QuotaIncreaseRequestIdentifier& identifier, const std::optional<uint64_t>& newQuota)
        : m_arguments(sessionID, origin, identifier, newQuota)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::ClientOrigin&, const WebKit::QuotaIncreaseRequestIdentifier&, const std::optional<uint64_t>&> m_arguments;
};

#if ENABLE(APP_BOUND_DOMAINS)
class HasAppBoundSession {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_HasAppBoundSession; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_HasAppBoundSessionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    explicit HasAppBoundSession(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};
#endif

#if ENABLE(APP_BOUND_DOMAINS)
class ClearAppBoundSession {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClearAppBoundSession; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClearAppBoundSessionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ClearAppBoundSession(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};
#endif

#if PLATFORM(IOS_FAMILY) && !PLATFORM(MACCATALYST)
class DisableServiceWorkerEntitlement {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_DisableServiceWorkerEntitlement; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

#if PLATFORM(IOS_FAMILY) && !PLATFORM(MACCATALYST)
class ClearServiceWorkerEntitlementOverride {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClearServiceWorkerEntitlementOverride; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClearServiceWorkerEntitlementOverrideReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};
#endif

class UpdateBundleIdentifier {
public:
    using Arguments = std::tuple<String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_UpdateBundleIdentifier; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_UpdateBundleIdentifierReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit UpdateBundleIdentifier(const String& bundleIdentifier)
        : m_arguments(bundleIdentifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const String&> m_arguments;
};

class ClearBundleIdentifier {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClearBundleIdentifier; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClearBundleIdentifierReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

#if PLATFORM(COCOA)
class AppPrivacyReportTestingData {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AppPrivacyReportTestingData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_AppPrivacyReportTestingDataReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<WebKit::AppPrivacyReportTestingData>;
    using Reply = CompletionHandler<void(WebKit::AppPrivacyReportTestingData&&)>;
    using Promise = WTF::NativePromise<WebKit::AppPrivacyReportTestingData, IPC::Error>;
    explicit AppPrivacyReportTestingData(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};
#endif

#if PLATFORM(COCOA)
class ClearAppPrivacyReportTestingData {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClearAppPrivacyReportTestingData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClearAppPrivacyReportTestingDataReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit ClearAppPrivacyReportTestingData(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};
#endif

class AddWebPageNetworkParameters {
public:
    using Arguments = std::tuple<PAL::SessionID, WebKit::WebPageProxyIdentifier, WebKit::WebPageNetworkParameters>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AddWebPageNetworkParameters; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    AddWebPageNetworkParameters(const PAL::SessionID& sessionID, const WebKit::WebPageProxyIdentifier& pageID, const WebKit::WebPageNetworkParameters& parameters)
        : m_arguments(sessionID, pageID, parameters)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebKit::WebPageProxyIdentifier&, const WebKit::WebPageNetworkParameters&> m_arguments;
};

class RemoveWebPageNetworkParameters {
public:
    using Arguments = std::tuple<PAL::SessionID, WebKit::WebPageProxyIdentifier>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_RemoveWebPageNetworkParameters; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    RemoveWebPageNetworkParameters(const PAL::SessionID& sessionID, const WebKit::WebPageProxyIdentifier& pageID)
        : m_arguments(sessionID, pageID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebKit::WebPageProxyIdentifier&> m_arguments;
};

class CountNonDefaultSessionSets {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_CountNonDefaultSessionSets; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_CountNonDefaultSessionSetsReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<size_t>;
    using Reply = CompletionHandler<void(size_t)>;
    using Promise = WTF::NativePromise<size_t, IPC::Error>;
    explicit CountNonDefaultSessionSets(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class AllowFileAccessFromWebProcess {
public:
    using Arguments = std::tuple<WebCore::ProcessIdentifier, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AllowFileAccessFromWebProcess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_AllowFileAccessFromWebProcessReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    AllowFileAccessFromWebProcess(const WebCore::ProcessIdentifier& processIdentifier, const String& url)
        : m_arguments(processIdentifier, url)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ProcessIdentifier&, const String&> m_arguments;
};

class AllowFilesAccessFromWebProcess {
public:
    using Arguments = std::tuple<WebCore::ProcessIdentifier, Vector<String>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AllowFilesAccessFromWebProcess; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = true;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_AllowFilesAccessFromWebProcessReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    AllowFilesAccessFromWebProcess(const WebCore::ProcessIdentifier& processIdentifier, const Vector<String>& urls)
        : m_arguments(processIdentifier, urls)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ProcessIdentifier&, const Vector<String>&> m_arguments;
};

class TerminateRemoteWorkerContextConnectionWhenPossible {
public:
    using Arguments = std::tuple<WebKit::RemoteWorkerType, PAL::SessionID, WebCore::RegistrableDomain, WebCore::ProcessIdentifier>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_TerminateRemoteWorkerContextConnectionWhenPossible; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    TerminateRemoteWorkerContextConnectionWhenPossible(WebKit::RemoteWorkerType workerType, const PAL::SessionID& sessionID, const WebCore::RegistrableDomain& registrableDomain, const WebCore::ProcessIdentifier& processIdentifier)
        : m_arguments(workerType, sessionID, registrableDomain, processIdentifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<WebKit::RemoteWorkerType, const PAL::SessionID&, const WebCore::RegistrableDomain&, const WebCore::ProcessIdentifier&> m_arguments;
};

class GetPendingPushMessage {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_GetPendingPushMessage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_GetPendingPushMessageReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<std::optional<WebKit::WebPushMessage>>;
    using Reply = CompletionHandler<void(std::optional<WebKit::WebPushMessage>&&)>;
    using Promise = WTF::NativePromise<std::optional<WebKit::WebPushMessage>, IPC::Error>;
    explicit GetPendingPushMessage(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class GetPendingPushMessages {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_GetPendingPushMessages; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_GetPendingPushMessagesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<Vector<WebKit::WebPushMessage>>;
    using Reply = CompletionHandler<void(Vector<WebKit::WebPushMessage>&&)>;
    using Promise = WTF::NativePromise<Vector<WebKit::WebPushMessage>, IPC::Error>;
    explicit GetPendingPushMessages(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class ProcessPushMessage {
public:
    using Arguments = std::tuple<PAL::SessionID, WebKit::WebPushMessage, WebCore::PushPermissionState>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ProcessPushMessage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ProcessPushMessageReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool, std::optional<WebCore::NotificationPayload>>;
    using Reply = CompletionHandler<void(bool, std::optional<WebCore::NotificationPayload>&&)>;
    using Promise = WTF::NativePromise<std::tuple<bool, std::optional<WebCore::NotificationPayload>>, IPC::Error>;
    ProcessPushMessage(const PAL::SessionID& sessionID, const WebKit::WebPushMessage& pushMessage, WebCore::PushPermissionState pushPermissionState)
        : m_arguments(sessionID, pushMessage, pushPermissionState)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebKit::WebPushMessage&, WebCore::PushPermissionState> m_arguments;
};

class ProcessNotificationEvent {
public:
    using Arguments = std::tuple<WebCore::NotificationData, WebCore::NotificationEventType>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ProcessNotificationEvent; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ProcessNotificationEventReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    ProcessNotificationEvent(const WebCore::NotificationData& data, WebCore::NotificationEventType eventType)
        : m_arguments(data, eventType)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::NotificationData&, WebCore::NotificationEventType> m_arguments;
};

class GetAllBackgroundFetchIdentifiers {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_GetAllBackgroundFetchIdentifiers; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_GetAllBackgroundFetchIdentifiersReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<Vector<String>>;
    using Reply = CompletionHandler<void(Vector<String>&&)>;
    using Promise = WTF::NativePromise<Vector<String>, IPC::Error>;
    explicit GetAllBackgroundFetchIdentifiers(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

class GetBackgroundFetchState {
public:
    using Arguments = std::tuple<PAL::SessionID, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_GetBackgroundFetchState; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_GetBackgroundFetchStateReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<std::optional<WebKit::BackgroundFetchState>>;
    using Reply = CompletionHandler<void(std::optional<WebKit::BackgroundFetchState>&&)>;
    using Promise = WTF::NativePromise<std::optional<WebKit::BackgroundFetchState>, IPC::Error>;
    GetBackgroundFetchState(const PAL::SessionID& sessionID, const String& identifier)
        : m_arguments(sessionID, identifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const String&> m_arguments;
};

class AbortBackgroundFetch {
public:
    using Arguments = std::tuple<PAL::SessionID, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_AbortBackgroundFetch; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_AbortBackgroundFetchReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    AbortBackgroundFetch(const PAL::SessionID& sessionID, const String& identifier)
        : m_arguments(sessionID, identifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const String&> m_arguments;
};

class PauseBackgroundFetch {
public:
    using Arguments = std::tuple<PAL::SessionID, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_PauseBackgroundFetch; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_PauseBackgroundFetchReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    PauseBackgroundFetch(const PAL::SessionID& sessionID, const String& identifier)
        : m_arguments(sessionID, identifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const String&> m_arguments;
};

class ResumeBackgroundFetch {
public:
    using Arguments = std::tuple<PAL::SessionID, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ResumeBackgroundFetch; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ResumeBackgroundFetchReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    ResumeBackgroundFetch(const PAL::SessionID& sessionID, const String& identifier)
        : m_arguments(sessionID, identifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const String&> m_arguments;
};

class ClickBackgroundFetch {
public:
    using Arguments = std::tuple<PAL::SessionID, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClickBackgroundFetch; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_ClickBackgroundFetchReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    ClickBackgroundFetch(const PAL::SessionID& sessionID, const String& identifier)
        : m_arguments(sessionID, identifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const String&> m_arguments;
};

class SetPushAndNotificationsEnabledForOrigin {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::SecurityOriginData, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPushAndNotificationsEnabledForOrigin; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_SetPushAndNotificationsEnabledForOriginReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SetPushAndNotificationsEnabledForOrigin(const PAL::SessionID& sessionID, const WebCore::SecurityOriginData& origin, bool enabled)
        : m_arguments(sessionID, origin, enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::SecurityOriginData&, bool> m_arguments;
};

class RemovePushSubscriptionsForOrigin {
public:
    using Arguments = std::tuple<PAL::SessionID, WebCore::SecurityOriginData>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_RemovePushSubscriptionsForOrigin; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_RemovePushSubscriptionsForOriginReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<unsigned>;
    using Reply = CompletionHandler<void(unsigned&&)>;
    using Promise = WTF::NativePromise<unsigned, IPC::Error>;
    RemovePushSubscriptionsForOrigin(const PAL::SessionID& sessionID, const WebCore::SecurityOriginData& origin)
        : m_arguments(sessionID, origin)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const WebCore::SecurityOriginData&> m_arguments;
};

class HasPushSubscriptionForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID, URL>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_HasPushSubscriptionForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_HasPushSubscriptionForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    HasPushSubscriptionForTesting(const PAL::SessionID& sessionID, const URL& scopeURL)
        : m_arguments(sessionID, scopeURL)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const URL&> m_arguments;
};

#if ENABLE(INSPECTOR_NETWORK_THROTTLING)
class SetEmulatedConditions {
public:
    using Arguments = std::tuple<PAL::SessionID, std::optional<int64_t>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetEmulatedConditions; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetEmulatedConditions(const PAL::SessionID& sessionID, const std::optional<int64_t>& bytesPerSecondLimit)
        : m_arguments(sessionID, bytesPerSecondLimit)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const std::optional<int64_t>&> m_arguments;
};
#endif

#if HAVE(NW_PROXY_CONFIG)
class ClearProxyConfigData {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_ClearProxyConfigData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit ClearProxyConfigData(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};
#endif

#if HAVE(NW_PROXY_CONFIG)
class SetProxyConfigData {
public:
    using Arguments = std::tuple<PAL::SessionID, Vector<std::pair<Vector<uint8_t>, WTF::UUID>>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetProxyConfigData; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetProxyConfigData(const PAL::SessionID& sessionID, const Vector<std::pair<Vector<uint8_t>, WTF::UUID>>& proxyConfigurations)
        : m_arguments(sessionID, proxyConfigurations)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const Vector<std::pair<Vector<uint8_t>, WTF::UUID>>&> m_arguments;
};
#endif

class SetStorageSiteValidationEnabled {
public:
    using Arguments = std::tuple<PAL::SessionID, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetStorageSiteValidationEnabled; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetStorageSiteValidationEnabled(const PAL::SessionID& sessionID, bool enabled)
        : m_arguments(sessionID, enabled)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, bool> m_arguments;
};

class SetPersistedDomains {
public:
    using Arguments = std::tuple<PAL::SessionID, HashSet<WebCore::RegistrableDomain>>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_SetPersistedDomains; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetPersistedDomains(const PAL::SessionID& sessionsID, const HashSet<WebCore::RegistrableDomain>& domains)
        : m_arguments(sessionsID, domains)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&, const HashSet<WebCore::RegistrableDomain>&> m_arguments;
};

class GetAppBadgeForTesting {
public:
    using Arguments = std::tuple<PAL::SessionID>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkProcess_GetAppBadgeForTesting; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkProcess_GetAppBadgeForTestingReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<std::optional<uint64_t>>;
    using Reply = CompletionHandler<void(std::optional<uint64_t>&&)>;
    using Promise = WTF::NativePromise<std::optional<uint64_t>, IPC::Error>;
    explicit GetAppBadgeForTesting(const PAL::SessionID& sessionID)
        : m_arguments(sessionID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const PAL::SessionID&> m_arguments;
};

} // namespace NetworkProcess
} // namespace Messages
