load ../lib/import

# Run the command in a new mount namespace with /tmp remounted
# read-only, but with $TMP (which might be under /tmp) still writable.
#
# Arguments are passed as arguments to `sudo`.
__withRoTmp() {
	local mount="mount -o bind ${TMP@Q}{,} && mount -o bind,remount,ro /tmp{,}"
	local env=(
		"DBSCRIPTS_CONFIG=${DBSCRIPTS_CONFIG}"
		"XDG_CONFIG_HOME=${XDG_CONFIG_HOME}"
	)
	sudo -- unshare -m -- sh -c "${mount} && sudo -u ${USER@Q} ${env[*]@Q} \$@" -- "$@"
}

__db-import-pkg() {
	local ret=0
	# Since common.bash->config.local sets TMPDIR=${TMP}/tmp,
	# TMPDIR is necessarily != /tmp.
	# Which means that if we try to write anything directly under /tmp,
	# then we are erroneously disregarding TMPDIR.
	# So, make /tmp read-only to make that be an error.
	__withRoTmp db-import-pkg "$@" || ret=$?
	# Verify that it cleaned up after itself and TMPDIR is empty
	find "$TMPDIR" -mindepth 1 | diff - /dev/null
	return $ret
}

######################################################################

@test "import no blacklisted packages (x86_64)" {
	__releaseImportedPackage slavery      x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
	__doesNotExist "$TMP"/ftp/{core/os/x86_64,pool/packages,sources/packages}/slavery-*
}

@test "import no blacklisted packages (i686)" {
	__releaseImportedPackage slavery      i686 "$TMP/rsyncd/archlinux32/i686/core/core.db.tar.gz" "$TMP/rsyncd/archlinux32/pool"
	__releaseImportedPackage pkg-simple-c i686 "$TMP/rsyncd/archlinux32/i686/core/core.db.tar.gz" "$TMP/rsyncd/archlinux32/pool"

	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinux32" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/i686/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/i686/pkg-simple-c-1-1-i686.pkg.tar.xz" "$TMP/ftp/pool/archlinux32/pkg-simple-c-1-1-i686.pkg.tar.xz"
	__doesNotExist "$TMP"/ftp/{core/os/i686,pool/archlinux32,sources/archlinux32}/slavery-*
}

@test "import no blacklisted packages (armv7h)" {
	__releaseImportedPackage slavery      armv7h "$TMP/rsyncd/archlinuxarm/armv7h/core/core.db.tar.gz"
	__releaseImportedPackage pkg-simple-c armv7h "$TMP/rsyncd/archlinuxarm/armv7h/core/core.db.tar.gz"

	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinuxarm" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/armv7h/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/armv7h/pkg-simple-c-1-1-armv7h.pkg.tar.xz" "$TMP/ftp/pool/alarm/pkg-simple-c-1-1-armv7h.pkg.tar.xz"
	__doesNotExist "$TMP"/ftp/{core/os/alarm,pool/alarm,sources/alarm}/slavery-*
}

@test "import DBs with no blacklisted packages" {
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
}

@test "import updated packages" {
	__releaseImportedPackage slavery      x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"

	__updateImportedPackage pkg-simple-c
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-2/desc
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-2-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-2-x86_64.pkg.tar.xz"
}

@test "import .db files as 0664 (x86_64)" {
	__releaseImportedPackage slavery      x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
	__doesNotExist "$TMP"/ftp/{core/os/x86_64,pool/packages,sources/packages}/slavery-*
	[[ "$(stat -c '%a' -- "$TMP/ftp/core/os/x86_64/core.db.tar.gz")" = 664 ]]
}

@test "import .db files as 0664 (i686)" {
	__releaseImportedPackage slavery      i686 "$TMP/rsyncd/archlinux32/i686/core/core.db.tar.gz" "$TMP/rsyncd/archlinux32/pool"
	__releaseImportedPackage pkg-simple-c i686 "$TMP/rsyncd/archlinux32/i686/core/core.db.tar.gz" "$TMP/rsyncd/archlinux32/pool"

	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinux32" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/i686/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/i686/pkg-simple-c-1-1-i686.pkg.tar.xz" "$TMP/ftp/pool/archlinux32/pkg-simple-c-1-1-i686.pkg.tar.xz"
	__doesNotExist "$TMP"/ftp/{core/os/i686,pool/archlinux32,sources/archlinux32}/slavery-*
	stat -- "$TMP/ftp/core/os/i686/core.db.tar.gz"
	[[ "$(stat -c '%a' -- "$TMP/ftp/core/os/i686/core.db.tar.gz")" = 664 ]]
}

@test "import .db files as 0664 (armv7h)" {
	__releaseImportedPackage slavery      armv7h "$TMP/rsyncd/archlinuxarm/armv7h/core/core.db.tar.gz"
	__releaseImportedPackage pkg-simple-c armv7h "$TMP/rsyncd/archlinuxarm/armv7h/core/core.db.tar.gz"

	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinuxarm" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/armv7h/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/armv7h/pkg-simple-c-1-1-armv7h.pkg.tar.xz" "$TMP/ftp/pool/alarm/pkg-simple-c-1-1-armv7h.pkg.tar.xz"
	__doesNotExist "$TMP"/ftp/{core/os/armv7h,pool/alarm,sources/alarm}/slavery-*
	stat -- "$TMP/ftp/core/os/armv7h/core.db.tar.gz"
	[[ "$(stat -c '%a' -- "$TMP/ftp/core/os/armv7h/core.db.tar.gz")" = 664 ]]
}

@test "import fully-masked upstream" {
	__releaseImportedPackage pkg-any-a x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"
	__releaseImportedPackage pkg-any-a i686   "$TMP/rsyncd/archlinux32/i686/core/core.db.tar.gz"    "$TMP/rsyncd/archlinux32/pool"
	__releaseImportedPackage pkg-any-a armv7h "$TMP/rsyncd/archlinuxarm/armv7h/core/core.db.tar.gz"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg
	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinux32" __db-import-pkg
	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinuxarm" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-any-a-1-1/desc
	bsdtar xfO "$TMP/ftp/core/os/i686/core.db.tar.gz"   pkg-any-a-1-1/desc
	bsdtar xfO "$TMP/ftp/core/os/armv7h/core.db.tar.gz" pkg-any-a-1-1/desc

	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-any-a-1-1-any.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-any-a-1-1-any.pkg.tar.xz"
	__isLinkTo "$TMP/ftp/core/os/i686/pkg-any-a-1-1-any.pkg.tar.xz"   "$TMP/ftp/pool/packages/pkg-any-a-1-1-any.pkg.tar.xz"
	__isLinkTo "$TMP/ftp/core/os/armv7h/pkg-any-a-1-1-any.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-any-a-1-1-any.pkg.tar.xz"
}

@test "import removes dropped packages" {
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc

	repo-remove -q "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" pkg-simple-c

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	if bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc; then
		return 1
	fi
}

@test "import removes blacklisted packages" {
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc >/dev/null

	echo 'pkg-simple-c::::' >> "$TMP/httpd/blacklist.txt"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	if bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc >/dev/null; then
		return 1
	fi
}

@test "import moves packages" {
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/testing/os/x86_64/testing.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	__isLinkTo "$TMP/ftp/testing/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
	bsdtar xfO "$TMP/ftp/testing/os/x86_64/testing.db.tar.gz" pkg-simple-c-1-1/desc >/dev/null

	repo-remove -q "$TMP/rsyncd/archlinux/testing/os/x86_64/testing.db.tar.gz" pkg-simple-c
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc >/dev/null
	if bsdtar xfO "$TMP/ftp/testing/os/x86_64/testing.db.tar.gz" pkg-simple-c-1-1/desc >/dev/null; then
		return 1
	fi
}

@test "import arch=any packages with sub-pkgrel" {
	# This is modeled after the situation with 'asp' and 'asp32'

	__releaseImportedPackage pkg-any64 x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"
	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-any-2-1-any.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-any-2-1-any.pkg.tar.xz"
	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-any-2-1/desc

	__releaseImportedPackage pkg-any32 i686 "$TMP/rsyncd/archlinux32/i686/core/core.db.tar.gz" "$TMP/rsyncd/archlinux32/pool"
	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinux32" __db-import-pkg
	__isLinkTo "$TMP/ftp/core/os/i686/pkg-any-1-1.2-any.pkg.tar.xz" "$TMP/ftp/pool/archlinux32/pkg-any-1-1.2-any.pkg.tar.xz"
	bsdtar xfO "$TMP/ftp/core/os/i686/core.db.tar.gz" pkg-any-1-1.2/desc
}

@test "import respects INHERIT precedence" {
	__releaseImportedPackage pkg-any-a armv7h "$TMP/rsyncd/archlinuxarm/armv7h/core/core.db.tar.gz"
	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinux32" __db-import-pkg

	__releaseImportedPackage pkg-any-a x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"
	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg
	DBSCRIPTS_CONFIG="${TMP}/config.local.archlinuxarm" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-any-a-1-1/desc
	bsdtar xfO "$TMP/ftp/core/os/armv7h/core.db.tar.gz" pkg-any-a-1-1/desc

	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-any-a-1-1-any.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-any-a-1-1-any.pkg.tar.xz"
	__isLinkTo "$TMP/ftp/core/os/armv7h/pkg-any-a-1-1-any.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-any-a-1-1-any.pkg.tar.xz"
}

@test "import doesn't backdate packages" {
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"

	touch "$TMP/stamp"
	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg

	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
	[[ "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz" -nt "$TMP/stamp" ]]
	[[ "$TMP/ftp/lastupdate" -nt "$TMP/stamp" ]]
}

@test "import previously removed packages" {
	# Import the package once
	__releaseImportedPackage pkg-simple-c x86_64 "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz" "$TMP/rsyncd/archlinux/pool/packages"
	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg
	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"

	# Remove it
	mv -fT -- "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz"{,.old}
	:> "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz"
	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg
	if bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc; then return 1; fi

	# And add it again
	mv -fT -- "$TMP/rsyncd/archlinux/core/os/x86_64/core.db.tar.gz"{.old,}
	DBSCRIPTS_CONFIG="${TMP}/config.local.packages" __db-import-pkg
	bsdtar xfO "$TMP/ftp/core/os/x86_64/core.db.tar.gz" pkg-simple-c-1-1/desc
	__isLinkTo "$TMP/ftp/core/os/x86_64/pkg-simple-c-1-1-x86_64.pkg.tar.xz" "$TMP/ftp/pool/packages/pkg-simple-c-1-1-x86_64.pkg.tar.xz"
}
