/* wptKeyserver.cpp - W32 Keyserver Access
 *	Copyright (C) 2000-2005 Timo Schulz
 *	Copyright (C) 2001 Marco Cunha
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <windows.h>
#include <stdio.h>
#include <sys/stat.h>
#include <ctype.h>

#include "wptKeyserver.h"
#include "wptErrors.h"
#include "wptTypes.h"
#include "wptNLS.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptGPG.h"
#include "wptRegistry.h"

static char base64code[] =
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

keyserver server[MAX_KEYSERVERS] = {0};
static keyserver_proxy_ctx proxy = {0};
static const char * server_list[] = {
    "hkp://wwwkeys.nl.pgp.net",
    "hkp://wwwkeys.pl.pgp.net",
    "hkp://wwwkeys.at.pgp.net",
    "hkp://wwwkeys.ch.pgp.net",
    "hkp://wwwkeys.de.pgp.net",
    "hkp://wwwkeys.dk.pgp.net",
    "hkp://wwwkeys.cz.pgp.net",
    "hkp://wwwkeys.es.pgp.net",
    "hkp://wwwkeys.eu.pgp.net",
    "hkp://wwwkeys.uk.pgp.net",
    "hkp://wwwkeys.us.pgp.net",
    "hkp://gnv.us.ks.cryptnet.net",
    "hkp://subkeys.pgp.net",
    "ldap://keyserver.pgp.com",
    NULL
};
static char hkp_errmsg[1024];
static int hkp_err = 0;
static u32 conf_timestamp = 0;
char * default_keyserver = NULL;
unsigned short default_keyserver_port = 0;


static void
base64_encode( const char *inputstr, char *outputstr, int maxlen )
{	
    int index = 0, temp = 0, res = 0, i = 0, inputlen = 0, len = 0;
    
    inputlen = strlen (inputstr); /* fixme: check if len > maxlen */ 
    for (i = 0; i <inputlen; i++) {
	res = temp;
	res = (res << 8) | (inputstr[i] & 0xFF);
	switch (index++) {
	case 0: outputstr[len++] = base64code[res >> 2 & 0x3F]; res &= 0x3; break;            
	case 1:	outputstr[len++] = base64code[res >> 4 & 0x3F]; res &= 0xF; break;            
	case 2:	outputstr[len++] = base64code[res >> 6 & 0x3F];
	        outputstr[len++] = base64code[res & 0x3F]; res = index = 0; break;
	}
	temp = res;
    }
    
    switch( index ) {
    case 0: break;        
    case 2: outputstr[len++] = base64code[temp << 2 & 0x3F];	
	    outputstr[len++] = '='; break;        
    case 1: outputstr[len++] = base64code[temp << 4 & 0x3F];	
	    outputstr[len++] = '=';
	    outputstr[len++] = '=';
    }
    
    outputstr[len] = '\0';
} /* base64_encode */


static const char *
skip_type_prefix (const char * hostname)
{
    if (hostname && !strncmp (hostname, "http://", 7))
	hostname += 7;
    else if (hostname && !strncmp (hostname, "hkp://", 6)) 
	hostname += 6;
    else if (hostname && !strncmp (hostname, "finger://", 9))
	hostname += 9;
    else if (hostname && !strncmp (hostname, "ldap://", 7))
	hostname += 7;
    return hostname;
}


static int
check_hkp_response (const char *resp, int recv)
{
    int ec, len;
    char *p, * end;

    ec = recv ? WPTERR_WINSOCK_RECVKEY : WPTERR_WINSOCK_SENDKEY;
    if (!strstr (resp, "HTTP/1.0 200 OK") &&
	!strstr (resp, "HTTP/1.1 200 OK")) /* http error */
	return ec;
    if (strstr (resp, "Public Key Server -- Error")
	|| strstr (resp, "Public Key Server -- Error")
	|| strstr (resp, "No matching keys in database")) {
	p = strstr (resp, "<p>");
	if (p && strlen (p) < sizeof (hkp_errmsg)) {
	    end = strstr (p, "</p>");
	    len = end? (end - p + 1) : strlen (p);
	    strncpy (hkp_errmsg, p, len);
	    hkp_err = 1;
	}
	return ec;
    }
    return 0;
} /* check_hkp_response */


static int
sock_getline (int fd, char *buf, int buflen, int *nbytes)
{
    char ch;
    int count = 0;	
    
    if (nbytes)
	*nbytes = 0;
    *buf = 0;
    while (recv (fd, &ch, 1, 0) > 0) {
	*buf++ = ch;
	count++;
	if (ch == '\n' || count == (buflen - 1)) {
	    *buf = 0;
	    if (nbytes)
		*nbytes = count;
	    return 0;
	}
    }

    return -1;
} /* sock_getline */


static int
sock_select (int fd)
{
    FD_SET rfd;
    timeval tv = {1, 0};

    FD_ZERO (&rfd);
    FD_SET (fd, &rfd);
    if (select (fd + 1, &rfd, NULL, NULL, &tv) == SOCKET_ERROR)
	return SOCKET_ERROR;
    if (FD_ISSET (fd, &rfd))
	return 1;
    return 0;
} /* sock_select */


static int
sock_read( int fd, char *buf, int buflen, int *nbytes )
{	
    DWORD nread;
    int nleft = buflen;
    int rc, n = 0;

    while (nleft > 0) {
	if (n >= 10)
	    return WPTERR_WINSOCK_TIMEOUT;
	if ( (rc = sock_select( fd )) == SOCKET_ERROR )
	    return SOCKET_ERROR;
	else if( !rc )
	    n++;
	else {
	    nread = recv(fd, buf, nleft, 0);
	    if (nread == SOCKET_ERROR)
		return SOCKET_ERROR;
	    else if (!nread)
		break;
	    nleft -= nread;
	    buf += nread;
	}	
    }
    if (nbytes)
	*nbytes = buflen - nleft;
	
    return 0;
} /* sock_read */


static int
sock_write( int fd, const char *buf, int buflen )
{
    DWORD nwritten;
    int nleft = buflen;

    while (nleft > 0) {
	nwritten = send (fd, buf, nleft, 0);
	if (nwritten == SOCKET_ERROR)
	    return SOCKET_ERROR;
	nleft -= nwritten;
	buf += nwritten;	
    }

    return 0;
} /* sock_write */


void
set_default_kserver (void)
{
    char * p = get_reg_entry_keyserver ("Default");
    free_if_alloc (default_keyserver);
    default_keyserver = p && *p != ' ' ? p : m_strdup (DEF_HKP_KEYSERVER);
    p = get_reg_entry_keyserver ("Default_Port");
    if (p && *p) {
	default_keyserver_port = (u16)strtoul (p, NULL, 10);
	free_if_alloc (p);
    }
    else
	default_keyserver_port = HKP_PORT;
} /* set_default_kserver */


/*
 * Initialize the Winsock2 interface.
 */
int
wsock_init (void)
{
    WSADATA wsa;

    if (WSAStartup (0x202, &wsa))
	return WPTERR_WINSOCK_INIT;
    set_default_kserver ();
    return 0;
} /* wsock_init */


/*
 * Should be called after the keyserver access.
 */
void
wsock_end (void)
{
    int i;

    free_if_alloc (default_keyserver);
    default_keyserver = NULL;
    for (i=0; i < MAX_KEYSERVERS; i++) {
	if (server[i].used)
	    free_if_alloc (server[i].name);
    }
    WSACleanup ();
} /* wsock_end */


const char*
wsock_strerror( void )
{    
    static char buf[384];
    int ec = WSAGetLastError( );
    
    switch ( ec ) {
    case WSAENETDOWN: return _("The network subsystem has failed");    
    case WSAHOST_NOT_FOUND: return _("Authoritative Answer Host not found");	        
    case WSAETIMEDOUT: return _("The connection has been dropped because of a network failure");
    default: _snprintf( buf, sizeof (buf) -1, _("Unknown Winsock error ec=%d"),ec); return buf;
    }
    return NULL;
} /* wsock_strerror */


const char*
kserver_strerror( void )
{	
    if( hkp_err )
	return hkp_errmsg;
    return NULL;
} /* kserver_strerror */


const char *
kserver_get_hostname (int idx, int type, u16 *port)
{
    if (type == -1) {
	*port = default_keyserver_port;
	return default_keyserver;
    }
    else if (!type && idx < DIM (server_list)) {
	*port = HKP_PORT;
	return server_list[idx];
    }
    return NULL;
}


int
kserver_check_inet_connection (void)
{
    int fd;

    if (!kserver_connect (default_keyserver, default_keyserver_port, &fd)) {
	closesocket (fd);
	return 0;
    }
    return -1;
}


/*
 * If the request contains the keyid, it have to be
 * always postfix with '0x'+keyid. This code checks
 * if the keyid is a decimal value and if so if it contains
 * the '0x'. If not it is inserted.
 */
const char*
kserver_check_keyid (const char *keyid)
{	
    static char id[20];

    if (strstr (keyid, "@"))
	return keyid; /* email address */
    if (strncmp (keyid, "0x", 2)) {
	memset (&id, 0, sizeof (id));
	_snprintf (id, sizeof (id)-1, "0x%s", keyid);
	return id;
    }
    return keyid;
} /* kserver_check_keyid */


static void
kserver_update_proxyuser (const char *proxy_user, const char *proxy_pass)
{
    char t[257]; /* user:pass = 127+1+127+1 = 257 */

    free_if_alloc (proxy.base64_user);
    proxy.base64_user = new char[4*strlen( proxy_user ) / 3 + 32];
    if( !proxy.base64_user )
	BUG (0);
    _snprintf (t, sizeof (t)-1, "%s:%s", proxy_user, proxy_pass);
    base64_encode (t, proxy.base64_user, 257);
    free_if_alloc (proxy.user);
    free_if_alloc (proxy.pass);
    proxy.user = m_strdup (proxy_user);
    proxy.pass = m_strdup (proxy_pass);
} /* kserver_update_proxyuser */


static int
check_URL (const char * buf)
{
    if (strlen (buf) < 7)
	return -1;
    if( !strstr (buf, "ldap://") 
	&& !strstr (buf, "http://") 
	&& !strstr (buf, "finger://")
	&& !strstr (buf, "hkp://"))
	return -1;
    return 0;
} /* check_URL */
    

static int
proto_from_URL (const char * buf)
{
    if (strstr( buf, "ldap"))
	return KSPROTO_LDAP;
    else if (strstr( buf, "finger"))
	return KSPROTO_FINGER;
    return KSPROTO_HTTP;
} /* proto_from_URL */


void
keyserver_set_default (const char * hostname, u16 port)
{
    if (hostname) {
	free_if_alloc (default_keyserver);
	default_keyserver = m_strdup (hostname);
	if (!default_keyserver)
	    BUG (0);
	default_keyserver_port = port;
    }
    server[0].name =  default_keyserver;
    server[0].used = 1;
    server[0].port = port;
    server[0].proto = proto_from_URL (default_keyserver);
} /* keyserver_set_default */


static const char *
skip_whitespace (const char * str)
{
    while (str && *str)
    {
	if (*str == ' ' ||
	    *str == '\t' ||
	    *str == '\n' ||
	    *str == '\r')
	{
	    str++;
	    continue;
	}
	break;
    }
    return str;
}


int
kserver_load_conf (const char * conf)
{
    struct stat statbuf;
    FILE *fp;
    char buf[1024], * s, * p;
    char *user = NULL, *pass = NULL;    
    int pos, proxy_auth = 0;
    int no_config=0;
    
    for (pos = 0; pos < MAX_KEYSERVERS; pos++) {
	server[pos].used = 0;
	server[pos].port = HKP_PORT;
    }
    
    fp = fopen (conf, "rb");
    if (!fp) {
	for( pos = 0; server_list[pos]; pos++ ) {
	    server[pos].used = 1;
	    server[pos].name = (char *)server_list[pos];
	    server[pos].proto = proto_from_URL( server_list[pos] );
	}
	no_config=1;
    }
    get_reg_proxy_prefs( &proxy.host, &proxy.port, &user, &pass );
    if( user && pass )
	kserver_update_proxyuser( user, pass );	
    else if( user && !pass || !user && pass ) {
	msg_box( NULL, _("Invalid proxy configuration."
			 "You need to set a user and a password"
	                 "to use proxy authentication!"), _("Proxy Error"), MB_ERR );
    }
    /* check if the host has a http:// prefix and skip it */
    if( proxy.host && !strncmp( proxy.host, "http://", 7 ) ) {
	const char *s = proxy.host+7;
	char * p = m_strdup (s);
	if (!p)
	    BUG (0);
	free_if_alloc (proxy.host);
	proxy.host = p;
    }
    free_if_alloc (user);
    free_if_alloc (pass);
    
    pos = 0;
    while( fp && !feof( fp ) ) {
	s = fgets (buf, sizeof (buf)-1, fp);
	if (!s)
	    break;
	if (strstr (buf, "\r\n"))
	    buf[strlen (buf)-2] = '\0';
	if (strstr (buf, "\n"))
	    buf[strlen (buf)-1] = '\0';
	s = (char *)skip_whitespace (buf);
	if (*s == '#' || strlen (s) < 7)
	    continue;
	if (check_URL (s)) {
	    msg_box (NULL, _("All entries of this file must have a valid prefix.\n"
		             "Currently HKP/HTTP, LDAP and FINGER are supported.\n"), 
			     _("Keyserver Error"), MB_ERR);
	    continue;
	}
	p = strchr (s+6, ':');
	server[pos].used = 1;
	server[pos].proto = proto_from_URL (s);
	if (!p)
	    server[pos].name = m_strdup (s);
	else {	    
	    server[pos].port = atol (p+1);
	    if (server[pos].port < 0 || server[pos].port > 65536)
		server[pos].port = HKP_PORT;
	    server[pos].name = new char [(p-s)+2];
	    if (!server[pos].name)
		BUG (0);
	    memset (server[pos].name, 0, (p-s)+2);
	    memcpy (server[pos].name, s, (p-s));
	}
	pos++;
	if (pos > MAX_KEYSERVERS)
	{
	    msg_box (NULL, _("The keyserver limit is exceeded"), _("Keyserver Warning"), MB_INFO);
	    break;
	}
    }
    if (fp)
	fclose (fp);
    if (!pos) 
    {
	/* only issue an error if the config file exist but it has no valid entries. */
	keyserver_set_default (NULL, HKP_PORT);
	if (!no_config)
	    return WPTERR_CONFIG_FILE;	
    }

    if (!stat (conf, &statbuf))
	conf_timestamp = statbuf.st_mtime;
    return 0;
} /* kserver_load_conf */


const char *
kserver_get_proxy (int * r_port)
{
    if (proxy.host) 
    {
	if (r_port)
	    *r_port = proxy.port;
	return proxy.host;
    }
    return NULL;
} /* kserver_get_proxy */


const char *
kserver_get_proxy_info( int id )
{
    switch( id ) {   
    case PROXY_USER: return proxy.user;
    case PROXY_PASS: return proxy.pass;
    }
    return NULL;
} /* kserver_get_proxy_info */


/*
 * Connect to the keyserver 'hostname'.
 * We always use the HKP port.
 */
int
kserver_connect (const char *hostname, u16 port, int *conn_fd)
{
    int rc, fd;
    u32 iaddr;
    char host[128] = {0};
    struct hostent *hp;
    struct sockaddr_in sock;

    if (debug)
	log_f ("kserver_connect: %s:%d\r\n", hostname, port);
    if (!port)
	port = HKP_PORT;
    if (conn_fd)
	*conn_fd = 0;
    hostname = skip_type_prefix (hostname);
    
    memset (&sock, 0, sizeof (sock));
    sock.sin_family = AF_INET;
    sock.sin_port = proxy.host? htons (proxy.port) : htons (port);
    if (proxy.host)
	strncpy (host, proxy.host, 127);
    else
	strncpy (host, hostname, 127);
    
    if ((iaddr = inet_addr (host)) != INADDR_NONE)
	memcpy (&sock.sin_addr, &iaddr, sizeof (iaddr));
    else if ((hp = gethostbyname (host))) {
	if (hp->h_addrtype != AF_INET)
	    return WPTERR_WINSOCK_RESOLVE;
	else if (hp->h_length != 4)
	    return WPTERR_WINSOCK_RESOLVE;
	memcpy (&sock.sin_addr, hp->h_addr, hp->h_length);
    }
    else {
	if (debug)
	    log_f ("gethostbyname: failed.\r\n");
	return WPTERR_WINSOCK_RESOLVE;
    }
    fd = socket (AF_INET, SOCK_STREAM, 0);
    if (fd == INVALID_SOCKET)	
	return WPTERR_WINSOCK_SOCKET;
    rc = connect (fd, (struct sockaddr *) &sock, sizeof (sock));
    if (rc == SOCKET_ERROR) {
	if (debug)
	    log_f ("connect: failed.\r\n");
	closesocket(fd);
	return WPTERR_WINSOCK_CONNECT;
    }

    if (conn_fd)
	*conn_fd = fd;
    WSASetLastError(0);
    return 0;
} /* kserver_connect */


static char*
kserver_urlencode (const char *pubkey, int octets, int *newlen)
{
    char *p, numbuf[5];
    size_t j = 0;
    int i, size;
    
    p = new char [2*octets];
    if (!p)
	BUG (0);

    for (size=0, i=0; i<octets; i++) {
	if (isalnum (pubkey[i]) || pubkey[i] == '-') {
	    p[size] = pubkey[i];
	    size++;
	}
	else if (pubkey[i] == ' ') {
	    p[size] = '+';
	    size++;
	}
	else {
	    sprintf(numbuf, "%%%02X", pubkey[i]);
	    for (j = 0; j<strlen(numbuf); j++) {
		p[size] = numbuf[j];
		size++;	
	    }
	}
    }
    p[size] = '\0';
    *newlen = size;
    return p;
} /* kserver_urlencode */


/*
 * Format a request for the given keyid (send).
 */
static char*
kserver_send_request (const char *hostname, u16 port, const char *pubkey, int octets)
{
    char *request = NULL, *enc_pubkey = NULL;
    int reqlen, enc_octets;
    
    if (debug)
        log_f ("kserver_send_request: %s:%d\n", hostname, port);
    if (!port)
	port = HKP_PORT;
    reqlen = 512 + strlen (hostname) + 2*strlen (pubkey);
    request = new char[reqlen];
    if (!request)
	BUG (0);
    enc_pubkey = kserver_urlencode (pubkey, octets, &enc_octets);
    if (!enc_pubkey || !enc_octets) {
	free_if_alloc (request);
        return NULL;
    }
    
    if (proxy.user) {
        _snprintf (request, reqlen-1,
                   "POST http://%s:%d/pks/add HTTP/1.0\r\n"
                   "Referer: \r\n"
                   "User-Agent: WinPT/W32\r\n"
                   "Host: %s:%d\r\n"
                   "Proxy-Authorization: Basic %s\r\n"
                   "Content-type: application/x-www-form-urlencoded\r\n"
                   "Content-length: %d\r\n"
                   "\r\n"
                   "keytext=%s"
                   "\n",
                   skip_type_prefix (hostname), port, hostname, port, 
		   proxy.base64_user, enc_octets+9, enc_pubkey);
    }
    else {
        _snprintf (request, reqlen-1,
                   "POST /pks/add HTTP/1.0\r\n"
                   "Referer: \r\n"
                   "User-Agent: WinPT/W32\r\n"
                   "Host: %s:%d\r\n"
                   "Content-type: application/x-www-form-urlencoded\r\n"
                   "Content-length: %d\r\n"
                   "\r\n"
                   "keytext=%s"
                   "\n",
                   skip_type_prefix (hostname), port, enc_octets+9, enc_pubkey);
    }
    free_if_alloc (enc_pubkey);
    if (debug)
        log_f("%s\n", request);
    return request;
} /* kserver_send_request */


/*
 * Interface receiving a public key.
 */
int
kserver_recvkey (const char *hostname, u16 port, const char *keyid, char *key, int maxkeylen)
{	
    int rc, n;
    int conn_fd;
    char *request = NULL;
    
    if (!port)
	port = HKP_PORT;
    hkp_err = 0; /* reset */    
    rc = kserver_connect (hostname, port, &conn_fd);
    if (rc)
        goto leave;
    
    request = new char[300+1];
    if (!request)
	BUG (0);
    
    if (proxy.host && proxy.user) {
	_snprintf (request, 300,
	    "GET http://%s:%d/pks/lookup?op=get&search=%s HTTP/1.0\r\n"
	    "Proxy-Authorization: Basic %s\r\n\r\n",
	    skip_type_prefix (hostname), port, keyid, proxy.base64_user);
    }
    else if (proxy.host) {
	_snprintf (request, 300,
	    "GET http://%s:%d/pks/lookup?op=get&search=%s HTTP/1.0\r\n\r\n",
	    skip_type_prefix (hostname), port, keyid);
    }    
    else {
        _snprintf (request, 300,
	    "GET /pks/lookup?op=get&search=%s HTTP/1.0\r\n\r\n", keyid);
    }   
    if (debug)
        log_f ("%s\n", request);
    
    rc = sock_write (conn_fd, request, strlen (request));
    if (rc == SOCKET_ERROR) {
        rc = WPTERR_WINSOCK_RECVKEY;
        goto leave;
    }
    
    rc = sock_read( conn_fd, key, maxkeylen, &n );
    if( rc == SOCKET_ERROR ) {
        rc = WPTERR_WINSOCK_RECVKEY;
        goto leave;
    }
    
    if( debug )
        log_f("%s\n", key);
    
    rc = check_hkp_response( key, 1 );
    if( rc )
        goto leave;
    
    WSASetLastError( 0 );
    
leave:
    closesocket( conn_fd );
    free_if_alloc( request );
    return rc;
} /* kserver_recvkey */


/* 
 * Interface to send a public key.
 */
int
kserver_sendkey (const char *hostname, u16 port, const char *pubkey, int len )
{
    int n, rc;
    int conn_fd;
    char *request = NULL;
    char log[2048];
    
    hkp_err = 0; /* reset */
    rc = kserver_connect (hostname, port, &conn_fd);
    if (rc)
	goto leave;
    
    request = kserver_send_request (hostname, port, pubkey, len);
    if( request == NULL ) {
	rc = WPTERR_GENERAL;
	goto leave;	
    }
    
    rc = sock_write( conn_fd, request, lstrlen(request) );
    if( rc == SOCKET_ERROR ) {
	rc = WPTERR_WINSOCK_SENDKEY;
	goto leave;	
    }
    
    rc = sock_read( conn_fd, log, sizeof (log)-1, &n );
    if( rc == SOCKET_ERROR ) {
	rc = WPTERR_WINSOCK_SENDKEY;
	goto leave;
    }
    if (debug)
	log_f("%s\n", log);
    
    rc = check_hkp_response( log, 0 );
    if( rc )
	goto leave;
    
    WSASetLastError( 0 );
    
leave:
    closesocket( conn_fd );
    free_if_alloc( request );
    return rc;
} /* kserver_sendkey */


int
kserver_search_init (const char * hostname, u16 port, const char * keyid, int * conn_fd)
{
    int rc, sock_fd;
    char * request = NULL;
    
    rc = kserver_connect (hostname, port, &sock_fd);
    if (rc) {
        *conn_fd = 0;
        goto leave;
    }
    
    request = new char[300+1];
    if (!request)
	BUG (0);
    
    if (proxy.host && proxy.user) {
	_snprintf (request, 300,
	    "GET http://%s:%d/pks/lookup?op=index&search=%s HTTP/1.0\r\n"
	    "Proxy-Authorization: Basic %s\r\n\r\n",
	    skip_type_prefix (hostname), port, keyid, proxy.base64_user);
    }    
    else if (proxy.host) {
	_snprintf (request, 300,
	    "GET http://%s:%d/pks/lookup?op=index&search=%s HTTP/1.0\r\n\r\n",
	    skip_type_prefix (hostname), port, keyid);
    }
    else {
        _snprintf (request, 300,
		   "GET /pks/lookup?op=index&search=%s HTTP/1.0\r\n\r\n", keyid);
    }
    
    if (debug)
        log_f("%s\n", request);
    
    if (sock_write (sock_fd, request, strlen (request)) == SOCKET_ERROR) {
        rc = WPTERR_GENERAL;
        goto leave;
    }
    
    *conn_fd = sock_fd;
    
leave:
    free_if_alloc (request);
    return rc;
} /* kserver_search_init */


int
kserver_search_chkresp (int fd)
{
    char buf[128];
    int n=0;

    /* parse response 'HTTP/1.0 500 OK' */
    if (sock_getline (fd, buf, 127, &n))
	return WPTERR_KEYSERVER_NOTFOUND;
    if (strncmp (buf, "HTTP/1.", 7))
	return WPTERR_KEYSERVER_NOTFOUND;
    if (strncmp (buf+(8+1), "200", 3))
	return WPTERR_KEYSERVER_NOTFOUND;
    return 0;
} /* kserver_search_chkresp */


int
kserver_search (int fd, keyserver_key * key)
{
    char buf[1024], *p;
    int uidlen, nbytes, pos = 0;
    
    if (debug)
	log_f ("keyserver_search:\n");
    
    if (sock_getline (fd, buf, sizeof (buf) - 1, &nbytes))
	return WPTERR_GENERAL;
    
    if (debug)
	log_f ("%s\n", buf);
    
    if (!strncmp (buf, "pub", 3)) {
        key->bits = atol (buf+3);
        p = strchr (buf, '>');
	if (!p)
	    goto fail;
	pos = p - buf + 1;
        memcpy (key->keyid, buf + pos, 8);
        key->keyid[8] = '\0';
        p = strstr (buf, "</a>");
	if (!p)
	    goto fail;
	pos = p - buf + 5;
        memcpy (key->date, buf + pos, 10);
        key->date[10] = '\0';
        pos += 10;
	p = buf + pos + 1;
	while (p && *p != '>')
	    p++;
	p++;
	uidlen = strlen (p) - 10;
	memcpy (key->uid, p, uidlen);
	key->uid[uidlen] = '\0';
	strcat (key->uid, ">");
	p = strchr (key->uid, '&');
	if (p) {
	    key->uid[(p-key->uid)] = '<';
	    memmove (key->uid+(p-key->uid)+1, key->uid+(p-key->uid)+4, strlen (key->uid)-3);
	}
	return 0;
    }

fail:
    key->bits = 0;
    memset (key, 0, sizeof *key);
    return 0;
} /* kserver_search */


static int
add_node( keyserver_cfgfile *cfg, const char *entry )
{
    keyserver_node *node;
    char *p = NULL;
    
    p = strchr( entry, '=' );
    if( p == NULL )
	return WPTERR_GENERAL;	
    node = new keyserver_node;
    if( !node )
	BUG( NULL );
    
    memset( node, 0, sizeof *node );
    node->used = 1;
    
    node->host.used = 1;
    node->host.proto = proto_from_URL( entry );
    memcpy( node->host.name, entry+7, p-entry-7 );
    node->next = cfg->list;
    cfg->list = node;
    
    return 0;
} /* add_node */


void
kserver_change_proxy( keyserver_proxy_ctx *ctx )
{    
    proxy.port = ctx->port;
    free_if_alloc (proxy.host);
    proxy.host = ctx->host? m_strdup( ctx->host ) : NULL;
    free_if_alloc( proxy.pass );
    proxy.pass = ctx->pass? m_strdup( ctx->pass ) : NULL;
    free_if_alloc( proxy.user );
    proxy.user = ctx->user? m_strdup( ctx->user ) : NULL;
    set_reg_proxy_prefs( proxy.host, proxy.port, proxy.user, proxy.pass );
} /* kserver_change_proxy */


int
kserver_read_config( const char *fname, keyserver_cfgfile **ret_cfg )
{
    FILE *fp;
    keyserver_cfgfile *cfg;
    char buf[256];
    int rc = 0;
    
    *ret_cfg = NULL;
    fp = fopen( fname, "rb" );
    if( fp == NULL )
        return WPTERR_FILE_OPEN;
    
    cfg = new keyserver_cfgfile;
    if ( !cfg )
	BUG( NULL );
    memset( cfg, 0, sizeof *cfg );
    
    while ( !feof( fp ) ) {
        if ( fgets( buf, sizeof(buf)-1, fp ) == NULL )
            break;
        if ( *buf == '\r' || *buf == '\n' || *buf == '#' )
            continue;
        buf[strlen(buf) -2] = '\0';
        if ( !strncmp( buf, "use_proxy=", 10 ) ) {
            char *p = strchr(buf, ':');
            if (!p)
                continue;
            cfg->proxy.port = atol( buf+(p-buf+1) ); 
            buf[p-buf] = '\0';
            cfg->proxy.host = m_strdup( buf+10 );
        }
        else if ( !strncmp( buf, "proxy_user=", 11 ) )
            cfg->proxy.user = m_strdup( buf+11 );
        else if ( !strncmp( buf, "proxy_pass=", 11 ) )
            cfg->proxy.pass = m_strdup( buf+11 );
        if ( !strncmp( buf, "http://", 7 ) 
	    || !strncmp( buf, "hkp://", 6 )
	    || !strncmp( buf, "ldap://", 7 ) ) {
            add_node( cfg, buf );
            cfg->nlist++;
        }
    }
    
    fclose( fp );
    *ret_cfg = cfg;
    
    return rc;
} /* kserver_read_config */


int
kserver_write_config( const char * fname, keyserver_cfgfile * cfg )
{
    
    return 0;
}


void
kserver_cfgfile_release( keyserver_cfgfile *cfg )
{	
    keyserver_node *k, *k2;

    proxy_release( &cfg->proxy );
    for( k = cfg->list; cfg->nlist--; k = k2 ) {
	k2 = k->next;
	free_if_alloc( k );	
    }
    free_if_alloc( cfg );
} /* kserver_cfgfile_release */


void
proxy_release( keyserver_proxy_ctx *ctx )
{
    free_if_alloc( ctx->host );
    free_if_alloc( ctx->pass );
    free_if_alloc( ctx->user );
} /* proxy_release */


int 
ldap_recvkey (const char * host, const char * keyid, char * key, int maxkeylen)
{
    const char *s;
    char *ksprg = NULL, *p = NULL;
    char temp[512], outf[512];
    FILE * inp;
    int rc, start_key = 0;
    STARTUPINFO si;
    PROCESS_INFORMATION pi;

    p = get_gnupg_path ();
    ksprg = new char[strlen (p) + 1 + 128];
    if (!ksprg)
	BUG (0);
    strcpy (ksprg, p);
    strcat (ksprg, "\\");
    strcat (ksprg, "gpgkeys_ldap.exe");
    free_if_alloc (p);
    if (file_exist_check (ksprg)) {
	log_box ( "LDAP Keyserver Plugin", MB_ERR, "Could not find LDAP keyserver module!");
	rc = -1; 
	goto leave;
    }
    GetTempPath (sizeof (temp)-1, temp);
    strcpy (outf, temp);
    strcat (outf, keyid);
    strcat (outf, ".out");
    strcat (temp, keyid);
    inp = fopen (temp, "w+b");
    if( !inp ) {
	log_box ("LDAP Keyserver Plugin", MB_ERR, "%s: %s", temp,
		 winpt_strerror (WPTERR_FILE_OPEN));
	rc = -1;
	goto leave;
    }
    fprintf (inp, 
	"VERSION 0\n"
	"HOST %s\n"
	"OPTION verbose\n"
	"COMMAND GET\n"
	"\n"
	"%s\n", host? skip_type_prefix (host): "64.94.85.200", keyid);
    fclose (inp);

    memset (&si, 0, sizeof (si));
    si.cb = sizeof (si);
    si.dwFlags = STARTF_USESHOWWINDOW;
    si.wShowWindow = SW_HIDE;
    memset (&pi, 0, sizeof (pi));
    p = new char[strlen (ksprg) + strlen (temp) + strlen (outf) + 32];
    sprintf (p, "%s -o %s %s", ksprg, outf, temp);
    rc = CreateProcess (NULL, p, NULL, NULL, FALSE, 0, NULL, NULL, &si, &pi);
    if (!rc) {
	log_box ("LDAP Keyserver Plugin", MB_ERR, "Could not spawn process");
	rc = -1;
	goto leave;
    }
    CloseHandle (pi.hThread);
    WaitForSingleObject (pi.hProcess, INFINITE);
    CloseHandle (pi.hProcess);

    inp = fopen (outf, "rb");
    if (!inp) {
	log_box ("LDAP Keyserver Plugin", MB_ERR, "%s: %s", outf, 
		 winpt_strerror (WPTERR_FILE_OPEN));
	rc = -1;
	goto leave;
    }
    memset (key, 0, maxkeylen);
    while( !feof( inp ) ) {
	s = fgets( temp, sizeof (temp)-1, inp );
	if( !s )
	    break;
	if( !start_key && strstr( s, "KEY" ) && strstr( s, "BEGIN" ) ) {
	    start_key = 1;
	    continue;
	}	
	if( !start_key )
	    continue;
	strcat( key, temp );
	maxkeylen -= strlen( temp );
	if( maxkeylen < 0 || (strstr( s, "KEY" ) && strstr( s, "END" )) )	    
	    break;
    }
    fclose( inp );    

leave:
    if( !strlen( key ) )
	rc = WPTERR_WINSOCK_RECVKEY;
    free_if_alloc( p );
    free_if_alloc( ksprg );
    return rc;
} /* ldap_recvkey */


static int
finger_readline (int fd, char * buf, int nbytes)
{
    char c = 0;
    int n, pos = 0;
    
    while (nbytes > 0) {
        n = recv (fd, &c, 1, 0);
        if (n <= 0)
            break;
        if (c == '\n')
            break;
        buf[pos++] = c;
        nbytes--;
    }
    if (nbytes > 0)
        buf[pos++] = '\0';
    if (n <= 0)
        pos = n;
    return pos;
}


int
finger_recvkey (const char * host, const char * user, char * key, int maxkeylen)
{
    struct hostent * hp;
    struct sockaddr_in saddr;
    char buf[128];
    int fd, nread;
    int start_key = 0;
    int rc=0;

    fd = socket (PF_INET, SOCK_STREAM, 0);
    if (fd == -1)
        return WPTERR_WINSOCK_SOCKET;
    hp = gethostbyname (skip_type_prefix (host));
    if (!hp) {
	closesocket (fd);
        return WPTERR_WINSOCK_RESOLVE;
    }

    memset (&saddr, 0, sizeof (saddr));
    saddr.sin_family = AF_INET;
    saddr.sin_port = htons (FINGER_PORT);
    saddr.sin_addr = *(struct in_addr *)hp->h_addr;
    if (connect (fd, (struct sockaddr *)&saddr, sizeof (saddr))) {
        closesocket (fd);
        return WPTERR_WINSOCK_CONNECT;
    }
    send (fd, user, strlen (user), 0);
    send (fd, "\r\n", 2, 0);

    memset (key, 0, maxkeylen);
    while (maxkeylen > 0) {
        nread = finger_readline (fd, buf, sizeof (buf)-1);
        if (nread <= 0)
            break;
	strcat (buf, "\n");
        if (strstr (buf, "BEGIN PGP PUBLIC KEY BLOCK")) {
            strcat (key, buf);
            start_key = 1;
	    maxkeylen -= nread;
        }
        else if (strstr (buf, "END PGP PUBLIC KEY BLOCK" ) && start_key) {
	    strcat (key, buf);
	    start_key--;
	    maxkeylen -= nread;
            break;
        }
        else if (start_key) {
	    strcat (key, buf);
	    maxkeylen -= nread;
	}
    }
    closesocket (fd);
    if (start_key != 0)
	rc = WPTERR_WINSOCK_RECVKEY;
    return rc;
}
