/* editkey-util.c
 *      Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "gpgme-config.h"
#ifdef WITH_EDITKEY
#include "gpgme.h"
#include "context.h"
#include "ops.h"
#include "util.h"


gpgme_error_t
gpgme_editkey_new( gpgme_editkey_t * r_ctx )
{
    gpgme_editkey_t ctx;

    if( !r_ctx )
        return mk_error( Invalid_Value );
    *r_ctx = NULL;
    ctx = calloc( 1, sizeof *ctx );
    if( !ctx )
        return mk_error( Out_Of_Core );
    *r_ctx = ctx;

    return 0;
} /* gpgme_editkey_new */


static void
revoke_release (gpgme_editkey_t ctx)
{
    safe_free (ctx->u.revoke.reason_text);
}


static void
adduid_release (gpgme_editkey_t ctx)
{
    safe_free (ctx->u.adduid.comment);
    safe_free (ctx->u.adduid.email);
    safe_free (ctx->u.adduid.name);
}


void
gpgme_editkey_release( gpgme_editkey_t ctx )
{
    if (!ctx)
        return;

    switch (ctx->type) {
        case GPGME_EDITKEY_REVOKE: revoke_release (ctx); break;
        case GPGME_EDITKEY_ADDUID: adduid_release (ctx); break;
    }
    ctx->type = 0;
    safe_free (ctx);
} /* gpgme_editkey_release */


/**
 * DELUID
 **/
void
gpgme_editkey_deluid_set_id (gpgme_editkey_t ctx, int id)
{
    if (!ctx)
        return;

    ctx->u.deluid.id = id;
    ctx->type = GPGME_EDITKEY_DELUID;
} /* gpgme_editkey_deluid_set_uid */

/**
 * DELKEY
 **/
void
gpgme_editkey_delkey_set_id( gpgme_editkey_t ctx, int id )
{
    if( !ctx )
        return;
    ctx->u.delkey.id = id;
    ctx->type = GPGME_EDITKEY_DELKEY;
} /* gpgme_editkey_delkey_set_id */


/* XXX: remove the passphrase in the _set functions and have a global
        function to set it. */

/**
 * ADDUID
 **/
void
gpgme_editkey_adduid_set( gpgme_editkey_t ctx, const char *name,
                           const char *email, const char *comment,
                            const char *passwd )
{
    if( !ctx )
        return;

    /* name is a MUST field */
    ctx->u.adduid.name = strdup( name );
    ctx->u.adduid.email = strdup( email );
    if( !comment )
        ctx->u.adduid.use_comment = 0;
    else {
        ctx->u.adduid.comment = strdup( comment );
        ctx->u.adduid.use_comment = 1;
    }
    ctx->u.adduid.passwd = passwd;
    ctx->type = GPGME_EDITKEY_ADDUID;
} /* gpgme_editkey_adduid_set */


gpgme_error_t
gpgme_editkey_addrev_set (gpgme_editkey_t ctx, const char * userid,
                          const char * passwd)
{
    if (!ctx)
        return mk_error (Invalid_Value);
    if (!userid)
        return mk_error (Invalid_Mode);
    ctx->u.addrev.passwd = passwd;
    ctx->u.addrev.uid = userid;
    ctx->type = GPGME_EDITKEY_ADDREV;
    return 0;
} /* gpgme_editkey_addrev_set */


void
gpgme_editkey_addphoto_set( gpgme_editkey_t ctx, const char *jpegfile,
                             const char * passwd )
{
    if( !ctx )
        return;
    ctx->u.addphoto.jpg = jpegfile;
    ctx->u.addphoto.passwd = passwd;
    ctx->type = GPGME_EDITKEY_ADDPHOTO;
} /* gpgme_editkey_addphoto_set */


/**
 * TRUST
 **/
void
gpgme_editkey_trust_set( gpgme_editkey_t ctx, int val )
{
    if( !ctx )
        return;

    switch( val ) {
    case 0: ctx->u.trust.trust_val = NULL; break;
    case GPGME_TRUST_DONTKNOW: ctx->u.trust.trust_val = "1"; break;
    case GPGME_TRUST_NEVER:    ctx->u.trust.trust_val = "2"; break;
    case GPGME_TRUST_MARGINAL: ctx->u.trust.trust_val = "3"; break;
    case GPGME_TRUST_FULLY:    ctx->u.trust.trust_val = "4";break;
    case GPGME_TRUST_ULTIMATE: ctx->u.trust.trust_val = "5"; break;
    default:                   ctx->u.trust.trust_val = "1"; break;
    }

    ctx->type = GPGME_EDITKEY_TRUST;
} /* gpgme_editkey_trust_set */

/**
 * SIGN
 **/
gpgme_error_t
gpgme_editkey_sign_set (gpgme_editkey_t ctx,
                        const char * passwd,
                        int sigclass,
                        int sigtype,
                        const char * param)
{
    if (!ctx)
        return mk_error (Invalid_Value);
    if (sigclass < 0 || sigclass > 3)
        return mk_error (Invalid_Mode);
    switch (sigtype) {
    case GPGME_EDITKEY_SIGN:
    case GPGME_EDITKEY_TSIGN:
    case GPGME_EDITKEY_NRSIGN:
    case GPGME_EDITKEY_NRLSIGN:
    case GPGME_EDITKEY_LSIGN:
        break;
    default:
        return mk_error (Invalid_Mode);
    }
    ctx->type = sigtype;
    ctx->u.sign.passwd = passwd;
    ctx->u.sign.sig_class = sigclass;
    ctx->u.sign.exp_date = param;

    return 0;
} /* gpgme_editkey_sign_set */

/**
 * ADDKEY
 **/
gpgme_error_t
gpgme_editkey_addkey_set (gpgme_editkey_t ctx, const char *passwd,
                          int algo, int size, int valid)
{
    if (!ctx)
        return mk_error (Invalid_Value);
    if (algo < 2 || algo > 6)
        return mk_error (Invalid_Mode);
    if (algo == 2 && size > 1024)
        size = 1024;
    if (size > 4096)
        size = 4096;

    ctx->u.addkey.algo = algo;
    ctx->u.addkey.size = size;
    ctx->u.addkey.valid = valid;
    ctx->u.addkey.passwd = passwd;
    ctx->type = GPGME_EDITKEY_ADDKEY;

    return 0;
} /* gpgme_editkey_addkey_set */


/**
 * PASSWD
 **/
void
gpgme_editkey_passwd_set( gpgme_editkey_t ctx, const char * old_passwd,
                          const char * new_passwd, int allow_empty )
{
    if( !ctx )
        return;
    if( old_passwd && new_passwd ) {
        ctx->u.passwd.old_passwd = old_passwd;
        ctx->u.passwd.new_passwd = new_passwd;
    }
    else if( !old_passwd && new_passwd ) {
        /* this is the case when the key was unprotected before. then
           GPG only asks once for a passphrase. */
        ctx->u.passwd.old_passwd = new_passwd;
        ctx->u.passwd.new_passwd = NULL;
    }
    ctx->u.passwd.allow_empty = allow_empty;
    ctx->u.passwd.send_old = 0;
    ctx->type = GPGME_EDITKEY_PASSWD;
} /* gpgme_editkey_passwd_set */


/**
 * PRIMARY
 **/
void
gpgme_editkey_primary_set( gpgme_editkey_t ctx, int id, const char * passwd )
{
    if( !ctx )
        return;

    ctx->u.primary.id = id;
    ctx->u.primary.passwd = passwd;
    ctx->type = GPGME_EDITKEY_PRIMARY;
} /* gpgme_editkey_primary_set */


/**
 * EXPIRE
 **/
void
gpgme_editkey_expire_set( gpgme_editkey_t ctx, int id, int days, const char *date,
                          const char *passwd )
{
    if( !ctx )
        return;
    ctx->u.expire.id = id;
    if( days && !date )
        ctx->u.expire.days = days;
    else if( date && !days ) {
        ctx->u.expire.date = date;
        ctx->u.expire.days = 0;
    }
    ctx->u.expire.passwd = passwd;
    ctx->type = GPGME_EDITKEY_EXPIRE;
} /* gpgme_editkey_expire_set */


/**
 * REVSIG
 **/
void
gpgme_editkey_revsig_set( gpgme_editkey_t ctx, int id, const char *passwd )
{
    if( !ctx )
        return;
    ctx->u.revsig.id = id;
    ctx->u.revsig.passwd = passwd;
    ctx->type = GPGME_EDITKEY_REVSIG;
} /* gpgme_editkey_revsig_set */


/**
 * REVKEY
 **/
void
gpgme_editkey_revkey_set( gpgme_editkey_t ctx, int id, int reason,
                          const char * passwd )
{
    if( !ctx )
        return;
    ctx->u.revkey.id = id;
    ctx->u.revkey.reason = reason;
    ctx->u.revkey.passwd = passwd;
    ctx->type = GPGME_EDITKEY_REVKEY;
}

/**
 * DELSIG
 **/
void
gpgme_editkey_delsig_set (gpgme_editkey_t ctx, int uid, int signo)
{
    if (!ctx)
        return;
    ctx->u.delsig.currno = 0;
    ctx->u.delsig.signo = signo;
    ctx->u.delsig.uid = uid;
    ctx->type = GPGME_EDITKEY_DELSIG;
}


/**
 * ENABLE
 **/
void
gpgme_editkey_enable_set( gpgme_editkey_t ctx )
{
    if( ctx )
        ctx->type = GPGME_EDITKEY_ENABLE;
}


/**
 * DISABLE
 **/
void
gpgme_editkey_disable_set( gpgme_editkey_t ctx )
{
    if( ctx )
        ctx->type = GPGME_EDITKEY_DISABLE;
}


/**
 * SETPREF
 **/
void
gpgme_editkey_setpref_set (gpgme_editkey_t ctx, const char * new_prefs,
                           int uid_idx, const char * passwd)
{
    if (!ctx)
        return;
    ctx->u.pref.id = 0;
    ctx->u.pref.uid_idx = uid_idx;
    ctx->u.pref.passwd = passwd;
    ctx->u.pref.new_prefs = new_prefs;
    ctx->type = GPGME_EDITKEY_SETPREF;
}


/**
 * KEYSERVER
 **/
void
gpgme_editkey_keyserver_set (gpgme_editkey_t ctx, const char *url,
                             int uid_idx, const char * passwd)
{
    if (!ctx)
        return;
    ctx->u.keyserv.uid_idx = uid_idx;
    ctx->u.keyserv.url = url;
    ctx->u.keyserv.passwd = passwd;
    ctx->type = GPGME_EDITKEY_KEYSERV;
}

void
gpgme_editkey_make_invalid (gpgme_editkey_t ctx)
{
    if( !ctx )
        return;
    ctx->type = 0;
} /* gpgme_editkey_make_invalid */


int
gpgme_editkey_is_valid( gpgme_editkey_t ctx )
{
    return ctx && ctx->type > 0 ? 1 : 0;
} /* gpgme_editkey_is_valid */


int
gpgme_editkey_is_secret( gpgme_editkey_t ctx, int val )
{
    if( ctx && val ) {
        ctx->key_pair = val;
        return 0;
    }
    return ctx && ctx->key_pair;
} /* gpgme_editkey_is_secret */
#endif /* WITH_EDITKEY */