/* encrypt.c -  Encrypt functions
 *	Copyright (C) 2000, 2001 Werner Koch (dd9jn), g10 Code GmbH
 *	Copyright (C) 2002-2005 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "util.h"
#include "context.h"
#include "ops.h"


#define use_symmetric(rset) (!(rset) || !gpgme_recipients_count ((rset)))

struct recp_info_s {
    struct recp_info_s * next;
    gpgme_error_t code;
    char recp[1];
};

struct encrypt_result_s {
    int okay;
    int no_recp;
    int inv_recp;
    int file_start;
    int file_done;
    struct recp_info_s * inf;
};


static gpgme_error_t
create_result_struct( gpgme_ctx_t ctx )
{
    assert ( !ctx->result.encrypt );
    ctx->result.encrypt = calloc( 1, sizeof *ctx->result.encrypt );
    if( !ctx->result.encrypt )
        return mk_error( Out_Of_Core );
    ctx->result_type = RESULT_TYPE_ENCRYPT;
    return 0;
} /* create_result_struct */


static gpgme_error_t
add_recp_info( _encrypt_result_t res, int code, const char * name )
{
    struct recp_info_s * r;

    r = calloc( 1, sizeof * r + strlen( name ) + 1 );
    if( !r )
	return mk_error( Out_Of_Core );
    r->code = code;
    strcpy (r->recp , name);
    r->next = res->inf;
    res->inf = r;

    return 0;
} /* add_recp_info */


void
_gpgme_release_encrypt_result( _encrypt_result_t res )
{    
    struct recp_info_s * r, * r2;

    if( !res )
	return;
    r = res->inf;
    while( r ) {
	r2 = r->next;
	safe_free( r );
	r = r2;
    }
    safe_free( res->inf );
    safe_free( res );
} /* _gpgme_release_encrypt_result */


void
_gpgme_encrypt_add_cipher (gpgme_ctx_t ctx)
{
    if (ctx->cipher_algo == -1)
	return;
    _gpgme_gpg_add_arg (ctx->gpg, "--cipher-algo");
    switch (ctx->cipher_algo) {
    case GPGME_CIPHER_3DES:     _gpgme_gpg_add_arg (ctx->gpg, "3DES");break;
    case GPGME_CIPHER_CAST5:    _gpgme_gpg_add_arg (ctx->gpg, "CAST5");break;
    case GPGME_CIPHER_BLOWFISH: _gpgme_gpg_add_arg (ctx->gpg, "BLOWFISH");break;
    case GPGME_CIPHER_AES128:   _gpgme_gpg_add_arg (ctx->gpg, "AES");break;
    case GPGME_CIPHER_AES192:   _gpgme_gpg_add_arg (ctx->gpg, "AES192");break;
    case GPGME_CIPHER_AES256:   _gpgme_gpg_add_arg (ctx->gpg, "AES256");break;
    case GPGME_CIPHER_TWOFISH:  _gpgme_gpg_add_arg (ctx->gpg, "TWOFISH");break;	
    default:                    _gpgme_gpg_add_arg (ctx->gpg, "AES");break;
    }
    if (ctx->s2k.used) {
	char buf[32];
	if (ctx->s2k.mode < 0 || ctx->s2k.mode > 3)
	    ctx->s2k.mode = GPGME_S2K_ITERSALTED;
	sprintf( buf, "%d", ctx->s2k.mode );
	_gpgme_gpg_add_arg (ctx->gpg, "--s2k-mode");
	_gpgme_gpg_add_arg (ctx->gpg,  buf);
	_gpgme_gpg_add_arg (ctx->gpg, "--s2k-digest-algo");
	switch (ctx->s2k.digest_algo) {
	case GPGME_MD_SHA1:  _gpgme_gpg_add_arg (ctx->gpg, "SHA1"); break;
	case GPGME_MD_RMD160:_gpgme_gpg_add_arg (ctx->gpg, "RIPEMD160"); break;
	default:             _gpgme_gpg_add_arg (ctx->gpg, "SHA1"); break;
	}
    }
} /* _gpgme_encrypt_add_cipher */


static void
encrypt_status_handler( gpgme_ctx_t ctx, gpg_statcode_t code, char * args )
{
    int i = 0;
    gpgme_error_t err;

    if (ctx->out_of_core)
        return;
    
    if( ctx->result_type == RESULT_TYPE_NONE ) {
        if( create_result_struct ( ctx ) ) {
            ctx->out_of_core = 1;
            return;
        }
    }
    assert( ctx->result_type == RESULT_TYPE_ENCRYPT );
    
    switch( code ) {
    case STATUS_END_ENCRYPTION:
	ctx->result.encrypt->okay = 1;
	break;

    case STATUS_NO_RECP:
        ctx->result.encrypt->no_recp = 1;
        break;
        
    case STATUS_INV_RECP:
	while( args[i] && args[i] != ' ' )
	    i++;
	while( args[i] && args[i] != ' ' )
	    i++;
	err = add_recp_info( ctx->result.encrypt, atol( args ), args + i );
	if( err )
	    ctx->out_of_core = 1;
	ctx->result.encrypt->inv_recp++;
        break;

    case STATUS_FILE_START:
	ctx->result.encrypt->file_start++;
	if( ctx->cb.interactiv )
	    ctx->cb.interactiv( ctx->cb.interactiv_value, code, NULL, args+2 );
	break;

    case STATUS_FILE_DONE:
	ctx->result.encrypt->file_done++;
	if( ctx->cb.interactiv )
	    ctx->cb.interactiv( ctx->cb.interactiv_value, code, NULL, NULL );
	break;

    case STATUS_PROGRESS:
	if (ctx->cb.progress)
	    _gpgme_progress_handler (ctx, args);
	break;
    }
} /* encrypt_status_handler */


static const char *
encrypt_command_handler (void * opaque, gpg_statcode_t code, const char * key)
{
    gpgme_ctx_t ctx = opaque;

    if (!ctx)
	return NULL;
    if (!strcmp (key, "untrusted_key.override"))
	return "N";
    if (!ctx->cb.interactiv)
	return NULL;
    if ((code == STATUS_GET_BOOL && !strcmp (key, "openfile.overwrite.okay"))
      ||(code == STATUS_GET_LINE && !strcmp (key, "openfile.askoutname" )) )
	return ctx->cb.interactiv (ctx->cb.interactiv_value, code, key, NULL);

    return NULL;
} /* encrypt_command_handler */


static gpgme_error_t
encrypt_start( gpgme_ctx_t ctx, gpgme_recipients_t recp, 
	       gpgme_data_t plain, gpgme_data_t ciph )
	      
{
    gpgme_error_t rc = 0;
    
    fail_on_pending_request (ctx);
    ctx->pending = 1;
    
    /* do some checks */    
    if( !gpgme_recipients_count ( recp ) ) {
        rc = mk_error( No_Recipients );
        goto leave;
    }
    
    /* create a process object */
    _gpgme_gpg_release (&ctx->gpg);
    rc = _gpgme_gpg_new (&ctx->gpg);
    if (rc)
        goto leave;
    
    _gpgme_gpg_set_status_handler (ctx->gpg, encrypt_status_handler, ctx);
    _gpgme_gpg_set_command_handler (ctx->gpg, encrypt_command_handler, ctx);
    
    /* build the commandline */
    _gpgme_gpg_add_arg( ctx->gpg, "--encrypt" );
    if( ctx->use_armor )
        _gpgme_gpg_add_arg( ctx->gpg, "--armor" );
    if (ctx->no_compress)
	_gpgme_gpg_add_arg (ctx->gpg, "-z 0");
    if (ctx->force_mdc)
	_gpgme_gpg_add_arg (ctx->gpg, "--force-mdc");
    _gpgme_add_comment (ctx);
    if (ctx->cb.progress)
	_gpgme_gpg_add_arg (ctx->gpg, "--enable-progress-filter");
    /* If we know that all recipients are valid (full or ultimate trust)
       we can pass suppress further checks */
    if( ctx->force_trust || _gpgme_recipients_all_valid( recp ) )
        _gpgme_gpg_add_arg( ctx->gpg, "--always-trust" );
    
    _gpgme_append_gpg_args_from_recipients( recp, ctx->gpg );
    if (ctx->use_logging)
	_gpgme_gpg_set_logging_handler (ctx->gpg, ctx);
    
    /* Check the supplied data */
    if( gpgme_data_get_type( plain ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }
    _gpgme_data_set_mode( plain, GPGME_DATA_MODE_OUT );
    if( !ciph || gpgme_data_get_type( ciph ) != GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( Invalid_Value );
        goto leave;
    }
    _gpgme_data_set_mode( ciph, GPGME_DATA_MODE_IN );
    /* Tell the gpg object about the data */
    if( ctx->use_tmpfiles ) {
	_gpgme_gpg_add_arg( ctx->gpg, "--yes" );
        _gpgme_gpg_add_arg( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg( ctx->gpg, _gpgme_get_tmpfile( 0 ) );
        _gpgme_data_write_to_tmpfile( plain );
        _gpgme_gpg_add_arg( ctx->gpg, _gpgme_get_tmpfile( 1 ) );
    }
    else {
        _gpgme_gpg_add_arg( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg( ctx->gpg, "-" );
        _gpgme_gpg_add_data( ctx->gpg, ciph, 1 );
        _gpgme_gpg_add_arg( ctx->gpg, "--" );
        _gpgme_gpg_add_data( ctx->gpg, plain, 0 );
    }
    
    /* and kick off the process */
    rc = _gpgme_gpg_spawn ( ctx->gpg, ctx );
    
leave:
    if( rc ) {
        ctx->pending = 0; 
        _gpgme_gpg_release( &ctx->gpg );
    }

    return rc;
} /* encrypt_start */


gpgme_error_t
file_encrypt_start( gpgme_ctx_t ctx, gpgme_recipients_t recp,
		    const char ** input, size_t nfiles, const char * output )
{
    gpgme_error_t rc = 0;
    int symmetric = 0;
    
    if( !input )
	return mk_error( Invalid_Value );

    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    if( use_symmetric( recp ) )
        symmetric = 1;
    
    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc ) {
        gpgme_release( ctx );
        return rc;
    }
        
    _gpgme_gpg_set_status_handler( ctx->gpg, encrypt_status_handler, ctx );
    if( ctx->cb.interactiv )
	_gpgme_gpg_set_command_handler( ctx->gpg, encrypt_command_handler, ctx );
    else
	_gpgme_gpg_add_arg (ctx->gpg, "--yes");
    if( ctx->use_armor )
        _gpgme_gpg_add_arg (ctx->gpg, "--armor");
    if (ctx->no_compress)
	_gpgme_gpg_add_arg (ctx->gpg, "-z 0");
    if (ctx->force_mdc)
	_gpgme_gpg_add_arg (ctx->gpg, "--force-mdc");
    _gpgme_add_comment (ctx);
    if( ctx->cb.progress )
	_gpgme_gpg_add_arg( ctx->gpg, "--enable-progress-filter" );
    if( !output )
	_gpgme_gpg_add_arg( ctx->gpg, "--no-mangle-dos-filenames" );

    if( symmetric ) {
        _gpgme_gpg_add_arg( ctx->gpg, "--symmetric" );
        _gpgme_encrypt_add_cipher( ctx );
        rc = _gpgme_add_passphrase( ctx );
        if( rc ) {
            _gpgme_gpg_release( &ctx->gpg );
            return rc;
        }
    }
    else {
        if( ctx->use_throwkeyid )
            _gpgme_gpg_add_arg( ctx->gpg, "--throw-keyid" );
        if( ctx->force_trust )
            _gpgme_gpg_add_arg( ctx->gpg, "--always-trust" );
	if( recp )
	    _gpgme_append_gpg_args_from_recipients( recp, ctx->gpg );
	if( ctx->pipemode || nfiles > 1 )
	    _gpgme_gpg_add_arg( ctx->gpg, "--encrypt-files" );
	else
	    _gpgme_gpg_add_arg( ctx->gpg, "--encrypt" );
	    
    }
    
    /* we cannot use --output for --encrypt-files */
    if( nfiles == 1  && !ctx->pipemode && output ) {
        _gpgme_gpg_add_arg( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg( ctx->gpg, output );
    }
    while( nfiles-- )
	_gpgme_gpg_add_arg( ctx->gpg, *input++ );
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }
    
    return rc;
} /* file_encrypt_start */


static gpgme_error_t
get_encrypt_result (gpgme_ctx_t ctx)
{
    struct encrypt_result_s * res;
    gpgme_error_t err = 0;
    int rc;

    assert (ctx->result.encrypt);
    res = ctx->result.encrypt;
    if (res->okay)
	return mk_error( No_Error );
    if (res->no_recp)
        err = mk_error (No_Recipients);
    else if (res->inv_recp)
	err = mk_error (Inv_Recipients);
    else if ((rc = gpgme_get_process_rc (ctx )) != 0) {
	DEBUG1 ("gpg return code=%d\n", rc);
	err = mk_error (Internal_GPG_Problem);
    }
    else if (!res->okay || (res->file_start != res->file_done))
	err = mk_error (Encryption_Failed);
    return err;
} /* get_encrypt_result */


/**
 * gpgme_op_encrypt:
 * @ctx: The context
 * @recp: A set of recipients 
 * @in: plaintext input
 * @out: ciphertext output
 * 
 * This function encrypts @in to @out for all recipients from
 * @recp.  Other parameters are take from the context @c.
 * The function does wait for the result.
 * 
 * Return value:  0 on success or an errorcode. 
 **/
gpgme_error_t
gpgme_op_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t recp, 
		  gpgme_data_t in, gpgme_data_t out )
                  
{
    gpgme_error_t err;
    
    err = encrypt_start( ctx, recp, in, out );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
        if( ctx->use_tmpfiles ) {
            _gpgme_data_read_from_tmpfile( out );
            _gpgme_del_tmpfiles( ctx->wipe_fnc );
        }
	err = get_encrypt_result( ctx );
    }

    return err;
} /* gpgme_op_encrypt */


gpgme_error_t
gpgme_op_file_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t rset, 
		       const char * input, const char * output )
{	
    gpgme_error_t err;
    const char * files[1];

    files[0] = input;
    err = file_encrypt_start( ctx, rset, files, 1, output );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
	err = get_encrypt_result( ctx );
    } 
    
    return err;
} /* gpgme_op_file_encrypt */


gpgme_error_t
gpgme_op_files_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t rset,
		        const char ** files, size_t nfiles )
{
    gpgme_error_t err;

    /* fixme: the result is actually only for the last file. we need
              a notification system which informs us with a callback
	      after each file is processed. */
    if( use_symmetric( rset ) ) {
	const char * s[1];
	size_t i;

	for( i=0; i < nfiles; i++ ) {
	    s[0] = files[i];
	    err = file_encrypt_start( ctx, NULL, s, 1, NULL );
	    if( !err ) {
		gpgme_wait( ctx, 1 );
		ctx->pending = 0;
		err = get_encrypt_result( ctx );
	    }
	    if( err )
		break;
	}
    }
    else {
	err = file_encrypt_start( ctx, rset, files, nfiles, NULL );
	if( !err ) {
	    gpgme_wait( ctx, 1 );
	    ctx->pending = 0;
	    err = get_encrypt_result( ctx );
	}
    }
    return err;
}


gpgme_error_t
gpgme_op_clip_encrypt( gpgme_recipients_t rset, int opts, gpgme_ctx_t *r_ctx )
{
    gpgme_error_t err;
    gpgme_ctx_t ctx = NULL;
    gpgme_data_t plain = NULL;
    gpgme_data_t ciph = NULL;
    
    err = gpgme_new( &ctx );
    if( err )
        return err;
    
    if( opts & GPGME_CTRL_TMPFILES )
        gpgme_control( ctx, GPGME_CTRL_TMPFILES, 1 );
    if( opts & GPGME_CTRL_FORCETRUST )
        gpgme_control( ctx, GPGME_CTRL_FORCETRUST, 1 );
    gpgme_control( ctx, GPGME_CTRL_ARMOR, 1 );	
    
    err = gpgme_data_new_from_clipboard (&plain);
    if( !err )
	err = gpgme_data_new( &ciph );
    if( !err )
	err = gpgme_op_encrypt( ctx, rset, plain, ciph );
    if( !err ) {
	gpgme_data_change_version( &ciph );
	gpgme_data_release_and_set_clipboard( ciph );
    }
    if( r_ctx )
	*r_ctx = ctx;
    else
	gpgme_release( ctx );
    gpgme_data_release( plain );
    return err;
} /* gpgme_op_clip_encrypt */


int
gpgme_recperr_count_items( gpgme_ctx_t ctx )
{
    struct recp_info_s * r;
    int ncount = 0;

    if( !ctx )
	return 0;
    if( ctx->result_type != RESULT_TYPE_ENCRYPT )
	return -1;
    for( r = ctx->result.encrypt->inf; r; r = r->next )
	ncount++;

    return ncount;
} /* gpgme_recperr_count_items */


const char*
gpgme_recperr_get( gpgme_ctx_t ctx, int idx, gpgme_error_t *r_code )
{
    struct recp_info_s * r;

    if( !ctx )
	return NULL;
    if( idx > gpgme_recperr_count_items( ctx ) )
	return NULL;
    for( r = ctx->result.encrypt->inf; r && idx--; r = r->next )
	;
    if( r ) {
	if( r_code )
	    *r_code = r->code;
	return r->recp;
    }

    return NULL;
} /* gpgme_recperr_get_code */
